#!/usr/bin/env bash
# This is where we keep whether or not to check the repo.
repo_check_global_variable=/home/akkadianuser/global_variables/akkadian_repo_check_is_enabled
# This is where we record whether or not the repo check failed.
repo_check_failed_global_variable=/home/akkadianuser/global_variables/akkadian_repo_check_failed

if [ ! -f $repo_check_global_variable ]; then
    echo "True" >$repo_check_global_variable
fi

# Check if the repo check global variable file exists and is readable.
if [[ -r $repo_check_global_variable ]]; then
    # If the file is readable, get its contents.
    repo_check_enabled=$(head -n 1 $repo_check_global_variable)
    # If the contents are equal to False, exit early and skip the connection test.
    if [[ "$repo_check_enabled" == "False" ]]; then
        echo "False" >$repo_check_failed_global_variable
        exit 0
    fi
fi

# Print the maria db rpm hashes to the log.
function print_maria_db_rpm_hashes() {
    if [[ -s /var/www/product_update/scripts/MariaDB-server-10.5.8-1.el7.centos.x86_64.rpm ]]; then
        md5sum /var/www/product_update/scripts/MariaDB-server-10.5.8-1.el7.centos.x86_64.rpm >>/var/www/product_update/update.log
    fi
    if [[ -s /var/www/product_update/scripts/galera-4-26.4.6-1.el7.centos.x86_64.rpm ]]; then
        md5sum /var/www/product_update/scripts/galera-4-26.4.6-1.el7.centos.x86_64.rpm >>/var/www/product_update/update.log
    fi
    if [[ -s /var/www/product_update/scripts/MariaDB-client-10.5.8-1.el7.centos.x86_64.rpm ]]; then
        md5sum /var/www/product_update/scripts/MariaDB-client-10.5.8-1.el7.centos.x86_64.rpm >>/var/www/product_update/update.log
    fi
    if [[ -s /var/www/product_update/scripts/MariaDB-shared-10.5.8-1.el7.centos.x86_64.rpm ]]; then
        md5sum /var/www/product_update/scripts/MariaDB-shared-10.5.8-1.el7.centos.x86_64.rpm >>/var/www/product_update/update.log
    fi
    if [[ -s /var/www/product_update/scripts/MariaDB-backup-10.5.8-1.el7.centos.x86_64.rpm ]]; then
        md5sum /var/www/product_update/scripts/MariaDB-backup-10.5.8-1.el7.centos.x86_64.rpm >>/var/www/product_update/update.log
    fi
    if [[ -s /var/www/product_update/scripts/MariaDB-common-10.5.8-1.el7.centos.x86_64.rpm ]]; then
        md5sum /var/www/product_update/scripts/MariaDB-common-10.5.8-1.el7.centos.x86_64.rpm >>/var/www/product_update/update.log
    fi
}

# Check that we can grab the contents of the the connection_test.info file at the Akkadian CentOS yum repo.
function validate_connection_to_repo() {
    if curl -k https://centosrepo.akkadianlabs.com/connection_test.info | grep -q "connection fine"; then
        echo "False" >$repo_check_failed_global_variable
        echo "Repository access confirmed." >>/var/www/product_update/update.log
    else
        echo "True" >$repo_check_failed_global_variable
        echo "Unable to reach repository." >>/var/www/product_update/update.log
        echo "500 - Error: The update requires the server to be able to reach centosrepo.akkadianlabs.com. Please refer to the release notes for more information." >/var/www/product_update/installation.progress
        sleep 5
        exit 1
    fi
}

function download_maria_db_rpms() {
    base="https://centosrepo.akkadianlabs.com/mariadb/"
    curl -k $base"MariaDB-server-10.5.8-1.el7.centos.x86_64.rpm" --output /var/www/product_update/scripts/MariaDB-server-10.5.8-1.el7.centos.x86_64.rpm
    curl -k $base"galera-4-26.4.6-1.el7.centos.x86_64.rpm" --output /var/www/product_update/scripts/galera-4-26.4.6-1.el7.centos.x86_64.rpm
    curl -k $base"MariaDB-client-10.5.8-1.el7.centos.x86_64.rpm" --output /var/www/product_update/scripts/MariaDB-client-10.5.8-1.el7.centos.x86_64.rpm
    curl -k $base"MariaDB-shared-10.5.8-1.el7.centos.x86_64.rpm" --output /var/www/product_update/scripts/MariaDB-shared-10.5.8-1.el7.centos.x86_64.rpm
    curl -k $base"MariaDB-backup-10.5.8-1.el7.centos.x86_64.rpm" --output /var/www/product_update/scripts/MariaDB-backup-10.5.8-1.el7.centos.x86_64.rpm
    curl -k $base"MariaDB-common-10.5.8-1.el7.centos.x86_64.rpm" --output /var/www/product_update/scripts/MariaDB-common-10.5.8-1.el7.centos.x86_64.rpm
    print_maria_db_rpm_hashes
}

function validate_mariadb_package_existence() {
    if md5sum -c /var/www/product_update/scripts/pre-installation-download-yum-md5.list; then
        echo "False" >$repo_check_failed_global_variable
        echo "Successfully downloaded yum update files." >>/var/www/product_update/update.log
    else
        echo "True" >$repo_check_failed_global_variable
        echo "Unable to download update files." >>/var/www/product_update/update.log
        echo "500 - Error: Unable to download yum update files. Please refer to the release notes for more information." >/var/www/product_update/installation.progress
        sleep 5
        exit 1
    fi
}

function check_open_ssl() {

    # 1.1.1c for ACM and 1.1.1d for APM
    if openssl version | grep -E "1.1.1c"; then
        if [[ -d /var/www/html/pme ]]; then
            echo "True" >$repo_check_failed_global_variable
            echo "Unable to validate OpenSSL version: $(openssl version)" >>/var/www/product_update/update.log
            echo "500 - Error: Incompatible OpenSSL version. An OVA migration will be required. Please reach out to support for more information." >/var/www/product_update/installation.progress
            sleep 5
            exit 1
        else
            echo "False" >$repo_check_failed_global_variable
            echo "OpenSSL version: $(openssl version)" >>/var/www/product_update/update.log
        fi
    elif openssl version | grep -E "1.1.1d"; then
        echo "False" >$repo_check_failed_global_variable
        echo "OpenSSL version: $(openssl version)" >>/var/www/product_update/update.log
    else
        if openssl version | grep -E "1.0.2k-fips"; then
            # it can be acm 3.3, then try new location since akkadian-executor does not have it
            if [[ -f /usr/local/openssl/bin/openssl ]]; then
                if /usr/local/openssl/bin/openssl version | grep "1.1.1c"; then
                    # we are ok, we need to proceed
                    echo "False" >$repo_check_failed_global_variable
                    echo "OpenSSL version: $(openssl version)" >>/var/www/product_update/update.log
                fi
            else
                echo "True" >$repo_check_failed_global_variable
                echo "Unable to validate OpenSSL version: $(openssl version)" >>/var/www/product_update/update.log
                echo "500 - Error: Incompatible OpenSSL version. An OVA migration will be required. Please reach out to support for more information." >/var/www/product_update/installation.progress
                sleep 5
                exit 1
            fi
        else
            echo "True" >$repo_check_failed_global_variable
            echo "Unable to validate OpenSSL version: $(openssl version)" >>/var/www/product_update/update.log
            echo "500 - Error: Incompatible OpenSSL version. An OVA migration will be required. Please reach out to support for more information." >/var/www/product_update/installation.progress
            sleep 5
            exit 1
        fi
    fi
}

function validate_cli_version() {
    current_cli_version=$(sh /home/akkadianuser/scripts/get_cli_version.sh)
    minimum_cli_version=3.3.0.281
    meets_or_exceeds_minimum=$(php -f /var/www/product_update/scripts/version_comparison.php "$current_cli_version" "$minimum_cli_version")
    if [[ $meets_or_exceeds_minimum == "True" ]]; then
        echo "False" >$repo_check_failed_global_variable
        echo "Current cli version $current_cli_version meets or exceeds the required version $minimum_cli_version." >>/var/www/product_update/update.log
    else
        echo "True" >$repo_check_failed_global_variable
        echo "The installed cli version $current_cli_version does not meet the required version $minimum_cli_version." >>/var/www/product_update/update.log
        echo "500 - Error: The installed CLI version ($current_cli_version) does not meet or exceed the required CLI version ($minimum_cli_version). Please update your CLI first and then attempt to install this update again." >/var/www/product_update/installation.progress
        sleep 5
        exit 1
    fi
}

# Need to connect to repo to get gcc, openssl-devel, bzip2-devel, zlib-devel, libffi-devel
if [[ ! -f /usr/local/bin/python3.7 ]]; then
    validate_connection_to_repo
fi

if [[ ! -d /opt/akkadian/aco-server/ ]]; then
    # Check if these have all been installed. If not, we need to connect to the repo and get them.
    installed_mariadb_server_version=$(rpm -q --qf "%{VERSION}" MariaDB-server)
    installed_galera_version=$(rpm -q --qf "%{VERSION}" galera-4)
    installed_mariadb_client_version=$(rpm -q --qf "%{VERSION}" MariaDB-client)
    installed_mariadb_shared_version=$(rpm -q --qf "%{VERSION}" MariaDB-shared)
    installed_mariadb_backup_version=$(rpm -q --qf "%{VERSION}" MariaDB-backup)
    installed_mariadb_common_version=$(rpm -q --qf "%{VERSION}" MariaDB-common)
    required_minimum_mariadb_server_version="10.5.8"
    required_minimum_galera_client_version="26.4.6"
    required_minimum_mariadb_client_version="10.5.8"
    required_minimum_mariadb_shared_version="10.5.8"
    required_minimum_mariadb_backup_version="10.5.8"
    required_minimum_mariadb_common_version="10.5.8"
    version_comparison="php -f /var/www/product_update/scripts/version_comparison.php"

    if [[
        $($version_comparison "$installed_mariadb_server_version" "$required_minimum_mariadb_server_version" "lt") == "True"
        || $($version_comparison "$installed_galera_version" "$required_minimum_galera_client_version" "lt") == "True"
        || $($version_comparison "$installed_mariadb_client_version" "$required_minimum_mariadb_client_version" "lt") == "True"
        || $($version_comparison "$installed_mariadb_shared_version" "$required_minimum_mariadb_shared_version" "lt") == "True"
        || $($version_comparison "$installed_mariadb_backup_version" "$required_minimum_mariadb_backup_version" "lt") == "True"
        || $($version_comparison "$installed_mariadb_common_version" "$required_minimum_mariadb_common_version" "lt") == "True"
    ]]; then
        echo "Unable to verify that galera and MariaDB RPMs were previously installed." >>/var/www/product_update/update.log
        validate_connection_to_repo
        download_maria_db_rpms
        validate_mariadb_package_existence
    fi

    # These checks should always happen.
    check_open_ssl
    validate_cli_version
fi

if [[ ! -d /opt/akkadian/aco-server/ ]]; then
    # ACM OVA has an issue where both 1.0.2k-fips and 1.1.1c are installed. We're going to skip the OpenSSL check for now.
    # TODO: Fix OpenSSL installation in ACM.
    if [[ ! -d /var/www/html/acm/ ]]; then
        check_open_ssl
    fi

    validate_cli_version
fi
