import subprocess
import time
from subprocess import PIPE, Popen

import click

from mongo_replication import remove_mongo_replication, add_mongo_replication, mongo_replication_postconditions

from network_management import get_ip, reach_ip, get_simple_hostname, is_ip, is_valid_hostname, get_domain, is_fqdn, \
    is_inside_domain, change_hostname
from src.cli_commons.commons_manager import boolean_evaluator, installed_application
from src.security_handler.security_handler import disable_ssh, disable_root_ssh_access, enable_ssh, \
    enable_root_ssh_access
from logs_handler import read_log

import configparser

mapper = configparser.RawConfigParser()
mapper.read('mapper.ini')

from email.message import EmailMessage
import smtplib
import ssl


def is_replication_mode_only(installed_app):
    if installed_app == 'aco':
        return check_only_replication_aco()
    elif installed_app == 'pme_cm':
        return check_only_replication_pme()
    else:
        return check_only_replication_acm()


def is_full_ha_mode(installed_app):
    if installed_app == 'aco':
        return check_full_ha_enable_aco()
    elif installed_app == 'pme_cm':
        return check_full_ha_enable_pme()
    else:
        return check_full_ha_enable_acm()


def remove_replication(installed_app):
    if installed_app == 'aco':
        remove_replication_aco()
    elif installed_app == 'pme_cm':
        remove_replication_pme()
    else:
        remove_replication_acm()

    click.pause("In order to fully disable HA, ensure all nodes in the cluster are off. Press any key to "
                "continue")


def maintenance_mode_prompt(installed_app):
    if click.confirm("Are you sure you want to enable maintenance mode, disabling database replication?"):
        maintenance_mode_on(installed_app)


def disable_replication_prompt(installed_app):
    if click.confirm("Are you sure you want to disable database replication?"):
        remove_replication(installed_app)


def maintenance_mode_on(installed_app):
    maintenance_info_save()
    if installed_app == 'aco':
        remove_replication_aco()
    elif installed_app == 'pme_cm':
        remove_replication_pme()
    else:
        remove_replication_acm()


def maintenance_mode_off(installed_app):
    replica_nodes = get_maintenance_mode_nodelist()
    write_nodes(replica_nodes)
    maintenance_info_remove()
    start_replication_main_process(installed_app, replica_nodes)


def start_full_ha(installed_app):
    click.clear()
    ha_nodes = []
    ip = get_simple_hostname().replace("\n", "")

    confirm = True
    nodes_numbers = 2

    max_nodes_number = calculate_nodes_amount(installed_app)

    if not click.confirm("In order to start Akkadian High Availability configuration, please ensure all nodes are "
                         "configured to be accessible from ssh and to have root remote access. This option "
                         "can be found in the Security main menu. Are you sure you want to continue?"):
        return
    else:
        click.secho('\nPlease follow the steps to configure High Availability Mode', fg='green')

        ha_nodes = get_ha_replication_nodes(confirm, ha_nodes, max_nodes_number, nodes_numbers, installed_app, 1)

        if ha_nodes is None:
            return False

        virtual_ip = get_vip()
        if virtual_ip is None:
            return False

        virtual_ip_mask = get_vmask()
        if virtual_ip_mask is None:
            return False

        password = get_cluster_password()

        show_cluster_detail(ha_nodes, ip, virtual_ip, virtual_ip_mask)

        if click.confirm('Do you want to save changes and enable Full High Availability?'):
            print("Please wait while High Availability is configured and enabled...")

            nodes = ""
            for i in ha_nodes:
                nodes += str(i) + " "

            install_and_configure_ha_processes(ip, nodes, password, virtual_ip, virtual_ip_mask)

            start_replication(installed_app, ha_nodes)
        else:
            return False
        return True


def start_replication(installed_app, ha_nodes):
    ip = get_simple_hostname().replace("\n", "")

    confirm = True
    nodes_numbers = 2

    max_nodes_number = calculate_nodes_amount(installed_app)

    if not ha_nodes:
        if not click.confirm("In order to start Akkadian High Availability configuration, please ensure all nodes are "
                             "configured to be accessible from ssh and has enable the root remote access. This option "
                             "can be found in the Security main menu. Are you sure you want to continue?"):
            return None

        ha_nodes = get_ha_replication_nodes(confirm, ha_nodes, max_nodes_number, nodes_numbers, installed_app, 2)

    if ha_nodes is None:
        return None

    click.secho("Configuring Replication, please wait...", fg='green')
    ha_nodes = [ip] + ha_nodes
    write_nodes(ha_nodes)

    r = start_replication_main_process(installed_app, ha_nodes)

    return r


def start_replication_main_process(installed_app, ha_nodes):
    try:
        if installed_app == 'aco':
            start_replication_aco(ha_nodes)
        elif installed_app == 'pme_cm':
            start_galera_replication()
        else:
            start_galera_replication()
        return True

    except Exception as e:
        return None


def open_ssh_and_root_access():
    enable_root_ssh_access()
    enable_ssh()


def start_replication_aco(ha_nodes):
    add_mongo_replication()
    if len(ha_nodes) < 3:
        mongo_replication_postconditions()


def start_replication_pme(ha_nodes):
    print('-- Adding Database Replication Settings')
    print('-- Resetting Synchronization Settings')
    remove_replication_general_configs()
    print('-- General Configs were removed')
    remove_replication_db_configs()
    print('-- Preparing Database Server for Replication')
    add_replication_preconditions()
    print('-- Activating Database Replication')
    determine_master_and_run_replication()
    print('-- Ending Database Server Configurations')
    add_replication_posconditions()


def start_galera_replication():
    p = Popen(["sh", mapper.get('GeneralScripts', 'enableReplicationGalera')], stdout=PIPE, stderr=PIPE)

    lines = []
    for line in p.stdout:
        line = line.decode("utf-8")
        lines.append(line)

    return lines


def calculate_nodes_amount(installed_app):
    if installed_app == "aco":
        max_nodes_number = 9
    else:
        max_nodes_number = 9
    return max_nodes_number


def install_and_configure_ha_processes(ip, nodes, password, virtual_ip, virtual_ip_mask):
    p = Popen(
        ["sh", mapper.get('GeneralScripts', 'haInstall'), ip, nodes, virtual_ip, virtual_ip_mask, password],
        stdout=PIPE, stderr=PIPE)
    out, err = p.communicate()
    time.sleep(10)
    for line in out.splitlines():
        line = line.decode("utf-8")
        print(line)
    for line in err.splitlines():
        line = line.decode("utf-8")
        print(line)

        if 'ERROR' in line:
            click.secho(line, fg='red')
            click.secho(
                'There was an error setting up HA. Check HA status and ensure hostnames are unique. You can validate this, using option 7 in Network menu')
            click.secho(
                'After ensuring hostnames are unique, you will need to remove cluster configurations. You can do this, using option 5 in High Availability menu')
            click.pause('Press any key to continue...')
    print('Please wait while High Availability is enabled...')
    time.sleep(20)


def show_cluster_detail(ha_nodes, ip, virtual_ip, virtual_ip_mask):
    click.secho('\nChanges to be saved:', fg='green')
    click.secho('Master node: %s' % ip, fg='green')
    for i in range(len(ha_nodes)):
        click.secho('Secondary node %s' % str(i + 1) + ': ' + ha_nodes[i], fg='green')
    click.secho('Virtual IP: %s' % virtual_ip, fg='green')
    click.secho('Virtual IP Mask: %s' % virtual_ip_mask, fg='green')


def get_cluster_password():
    password = click.prompt('Cluster Password', type=str, hide_input=True, confirmation_prompt=True)
    while '\\' in password or '/' in password or '$' in password:
        click.pause('Password cannot have \\ nor / nor $, please press any key to continue')
        password = click.prompt('Cluster Password', type=str, hide_input=True, confirmation_prompt=True)
    return password


def get_vmask():
    try:
        virtual_ip_mask = click.prompt('Virtual IP Mask, enter quit to exit', type=str).replace("\n", "")
        if virtual_ip_mask == 'quit':
            return None
    except click.Abort:
        return None

    return virtual_ip_mask


def get_vip():
    try:
        virtual_ip = click.prompt('Virtual IP, enter quit to exit', type=str).replace("\n", "")
        if virtual_ip == 'quit':
            return None
    except click.Abort:
        return None

    return virtual_ip




def get_ha_replication_nodes(confirm, ha_nodes, max_nodes_number, nodes_numbers, installed_app, type_server):

    if installed_app == 'aco':
        click.secho('We strongly recommend to setup a cluster with 3 or more nodes, please refer to the documentation',
                    fg='yellow')
    click.secho('Please when prompted, select the Domain Name or the FQDN (name.domain.com)', fg='yellow')
    click.secho('Node 1: %s' % get_simple_hostname(), fg='green')


    currenthostname = get_simple_hostname()
    currenthostname = currenthostname.strip()
    currenthostname = currenthostname.replace("\n", "")
    domain = get_domain(currenthostname)
    has_domain = (not domain == None)


    if (type_server == 1) and has_domain and not is_fqdn(currenthostname):
        click.pause('Current host name for Node 1 has invalid FQDN. Change it')
        return None

    ha_nodes = addNodes(nodes_numbers,max_nodes_number, domain, confirm, type_server,installed_app)

    return ha_nodes


def addNodes(nodes_numbers,max_nodes_number, domain, confirm, type_server,installed_app):
    switcher = {
        1: "first",
        2: "second",
        3: "third",
        4: "fourth",
        5: "fifth",
        6: "sixth",
        7: "seventh",
        8: "eighth",
        9: "ninth"
    }


    exit = 0
    currenthostname = get_simple_hostname()
    currenthostname = currenthostname.strip()
    currenthostname = currenthostname.replace("\n", "")
    ha_nodes = None
    has_domain = (not domain == None)

    while nodes_numbers <= max_nodes_number and confirm:

        # node 2 is mandatory to load
        if nodes_numbers <= 2 or  (nodes_numbers == 3 and installed_app == "aco"):
            confirm = True
            value = click.prompt('Please enter the %s hostname. Press quit to exit.\nNode %s' % (switcher.get(nodes_numbers), nodes_numbers) , type=str).replace("\n", "")

        # next node is not mandatory to load
        else:
            confirm = click.confirm('Do you want to add another Node?')
            if confirm:
                value = click.prompt('Please enter the %s hostname. Press quit to exit.\nNode %s' % (switcher.get(nodes_numbers), nodes_numbers) , type=str).replace("\n", "")


        if confirm:
            while value != 'quit'  and (( not reach_ip(value) or is_ip(value) ) or ( type_server == 1 and has_domain and not is_fqdn(value) ) or ( type_server == 1 and has_domain and not is_inside_domain(value,domain) ) or (not (ha_nodes is None) and any(value == s for s in ha_nodes)) or currenthostname == value ):

                if not reach_ip(value) or is_ip(value):
                    click.pause('Node is not reachable or you have entered an ip...')
                    click.secho('Node %s not reachable or you have entered an ip...' % value, fg='red')
                    if has_domain :
                        click.secho('Please enter the Domain Name or the FQDN (name.domain.com)', fg='yellow')
                    value = click.prompt('Please enter the %s hostname. Press quit to exit.\nNode %s' % (switcher.get(nodes_numbers), nodes_numbers) , type=str).replace("\n", "")

                elif type_server == 1 and has_domain and not is_fqdn(value):
                    click.secho('Host Domain Name for node %s is not valid: %s (name.domain.com)' % (nodes_numbers, value), fg='red')
                    value = click.prompt('Please enter the %s hostname. Press quit to exit.\nNode %s' % (switcher.get(nodes_numbers), nodes_numbers) , type=str).replace("\n", "")

                elif type_server == 1 and has_domain and not is_inside_domain(value,domain):
                    click.secho('Domain Name for node %s is diferent of %s (name.domain.com)' % (nodes_numbers, domain), fg='red')
                    value = click.prompt('Please enter the %s hostname. Press quit to exit.\nNode %s' % (switcher.get(nodes_numbers), nodes_numbers) , type=str).replace("\n", "")

                elif (not (ha_nodes is None) and any(value == s for s in ha_nodes)) or currenthostname == value:
                    click.secho('Host Domain Name for node %s is duplicated: %s (name.domain.com)' % (nodes_numbers, value), fg='red')
                    value = click.prompt('Please enter the %s hostname. Press quit to exit.\nNode %s' % (switcher.get(nodes_numbers), nodes_numbers) , type=str).replace("\n", "")



            if value == 'quit':
                if nodes_numbers == 2:
                    exit = 2
                else:
                    exit = 3
                return


            if ha_nodes is None:
                ha_nodes = [value]
            else:
                ha_nodes += [value]

            nodes_numbers += 1


    if exit == 2:
        return None
    else:
        return ha_nodes


def check_full_ha_enable_aco():
    return check_only_replication_aco() and are_pcs_pacemaker_corosync_running()


def check_full_ha_enable_pme():
    return check_only_replication_pme() and are_pcs_pacemaker_corosync_running()


def check_full_ha_enable_acm():
    return check_only_replication_acm() and are_pcs_pacemaker_corosync_running()


def check_only_replication_aco():
    return is_db_replication_set_in_mongo()


def check_only_replication_pme():
    return is_db_replication_set_in_galera()


def check_only_replication_acm():
    return is_db_replication_set_in_galera()


def remove_replication_aco():
    remove_mongo_replication()


def remove_replication_pme():
    remove_galera_replication()
    remove_replica_nodes_in_config()


def remove_replication_acm():
    remove_galera_replication()
    remove_replica_nodes_in_config()


def remove_galera_replication():
    try:
        p = Popen(["sh", mapper.get('GeneralScripts', 'removeReplicationGalera')], stdout=PIPE, stderr=PIPE)

        for line in p.stdout:
            line = line.decode("utf-8")
            print(line)

        click.pause("Press any key to continue...")
    except Exception as e:
        print(e)

    return


def remove_replica_nodes_in_config():
    try:
        p = Popen(["sh", mapper.get('GeneralScripts', 'removeReplicaNodes')], stdout=PIPE, stderr=PIPE)

    except Exception as e:
        print(e)
    return


def maintenance_info_save():
    try:
        p = Popen(["sh", mapper.get('GeneralScripts', 'maintenanceInfoSave')], stdout=PIPE, stderr=PIPE)

    except Exception as e:
        print(e)
    return


def maintenance_info_remove():
    try:
        p = Popen(["sh", mapper.get('GeneralScripts', 'maintenanceInfoRemove')], stdout=PIPE, stderr=PIPE)

    except Exception as e:
        print(e)
    return


def maintenance_is_mode_on():
    try:
        is_mode_on = False
        p = Popen(["sh", mapper.get('GeneralScripts', 'maintenanceIsModeOn')], stdout=PIPE, stderr=PIPE)
        for line in p.stdout:
            line = line.decode("utf-8")

        if "True" in line:
            is_mode_on = True
        else:
            is_mode_on = False

    except Exception as e:
        print(e)

    return is_mode_on


def remove_full_ha(installed_app):
    if click.confirm('Are You sure you want to remove the High Availability Configurations?'):
        disable_replication_prompt(installed_app)
        ha_disable_cluster()


def ha_disable_cluster():
    p = Popen(["sh", mapper.get('GeneralScripts', 'haDisableCluster')], stdout=PIPE, stderr=PIPE)
    for line in p.stdout:
        line = line.decode("utf-8")
        print(line)
    for line in p.stderr:
        print(line)
    p.wait()

    if p.returncode != 0:
        click.secho('[FAIL] Cannot reach high availability, is it running?', fg="red")
        return
    else:
        click.secho('[OK] High Availability is Disable', fg="yellow")

    time.sleep(3)
    click.pause('Please press any key to continue...')


########### LEGACY CODE ##############

def is_ha_enable():
    p = subprocess.Popen(["sh", mapper.get('GeneralScripts', 'isEnable')], stdout=PIPE,
                         stderr=PIPE)

    for line in p.stdout:
        line = line.decode("utf-8")
        if 'enable' in line:
            return True
        else:
            return False


def ha_enable():
    """
    Enable High Availability Service
    """
    print("Enabling...")
    p = Popen(["sh", mapper.get('GeneralScripts', 'haEnable')], stdout=PIPE, stderr=PIPE)
    for line in p.stdout:
        line = line.decode("utf-8")
        print(line)
    p.wait()

    if p.returncode != 0:
        click.secho('[FAIL] Cannot reach high availability, was it created?', fg="red")
    else:
        click.secho('[OK] High Availability is Enable', fg="yellow")

    time.sleep(3)


def disable_redundancy():
    try:

        if click.confirm("Are you sure you want to disable Master-Master Database Replication?"):
            p = Popen(["sh", mapper.get('GeneralScripts', 'disableRedundancy')], stdout=PIPE, stderr=PIPE)

            for line in p.stdout:
                line = line.decode("utf-8")
                print(line)

            click.pause("Press any key to continue...")
    except Exception as e:
        print(e)
    return


def ha_disable():
    """
    Disable High Availability Service
    """

    p = Popen(["sh", mapper.get('GeneralScripts', 'haDisable')], stdout=PIPE, stderr=PIPE)
    for line in p.stdout:
        line = line.decode("utf-8")
        print(line)
    for line in p.stderr:
        line = line.decode("utf-8")
        print(line)
    p.wait()

    if p.returncode != 0:
        click.secho('[FAIL] Cannot reach high availability, is it running?', fg="red")
    else:
        click.secho('[OK] High Availability is Disable', fg="yellow")

    time.sleep(3)


def add_node(ip):
    """
    Add High Availability Node
    """
    master = get_ip()
    p = subprocess.Popen(["sh", mapper.get('GeneralScripts', 'addNodeScript'), master, ip],
                         stdout=PIPE, stderr=PIPE)

    out, err = p.communicate()
    time.sleep(5)

    for line in out.splitlines():
        print(line)

    for line in err.splitlines():
        print(line)

        if 'Error' in line:
            click.pause('\nERROR: The new HA node was not added. Press any key to continue...')
            return

    click.pause('The new HA node was added correctly. Press any key to continue...')
    return


def remove_node(master, ip):
    """
    Remove High Availability Node
    """
    print("This will delay, please wait while everything necessary is configure...")
    p = subprocess.Popen(["sh", mapper.get('GeneralScripts', 'removeNodeScript'), master, ip],
                         stdout=PIPE, stderr=PIPE)

    out, err = p.communicate()
    time.sleep(10)

    for line in out.splitlines():
        print(line)

    for line in err.splitlines():
        print(line)

        if 'Error' in line:
            click.pause('ERROR: The HA node was not remove. Press any key to continue...')
            return

    click.pause('The new HA node was removed correctly. Press any key to continue...')
    return


def get_master():
    try:
        p = Popen(["sh", mapper.get('GeneralScripts', 'haGetMaster')], stdout=PIPE, stderr=PIPE)
        master = ''
        for line in p.stdout:
            line = line.decode("utf-8")
            master = line

        return master
    except:
        print('There was an error. Press any key to continue...')
        return 'No data'


def get_onlines():
    try:
        p = Popen(["sh", mapper.get('GeneralScripts', 'haGetOnlines')], stdout=PIPE, stderr=PIPE)
        out = ''
        nodes = ''
        for line in p.stdout:
            line = line.decode("utf-8")
            nodes = line

        nodes_space = nodes.split(" ")

        for node in nodes_space:
            if '[' not in node and ']' not in node:
                out = out + ' ' + node + '\n'

        return out
    except:
        print('There was an error. Press any key to continue...')
        return 'No data'


def get_offlines():
    try:
        p = Popen(["sh", mapper.get('GeneralScripts', 'haGetOfflines')], stdout=PIPE, stderr=PIPE)
        out = ''
        nodes = ''
        for line in p.stdout:
            line = line.decode("utf-8")
            nodes = line

        nodes_space = nodes.split(" ")

        for node in nodes_space:
            if '[' not in node and ']' not in node:
                out = out + ' ' + node + '\n'

        return out

    except:
        print('There was an error. Press any key to continue...')
        return 'No data'


def add_replication_preconditions():
    try:
        p = Popen(["sh", mapper.get('GeneralScripts', 'dbAddReplicationPreconditions')], stdout=PIPE, stderr=PIPE)

        lines = []
        for line in p.stdout:
            line = line.decode("utf-8")
            lines.append(line)

        print("Success")

        return
    except Exception as e:
        print(e)
        return


def add_replication_posconditions():
    try:
        p = Popen(["sh", mapper.get('GeneralScripts', 'dbAddReplicationPosconditions')], stdout=PIPE, stderr=PIPE)

        lines = []
        for line in p.stdout:
            line = line.decode("utf-8")
            lines.append(line)

        print("Success")

        return
    except Exception as e:
        print(e)
        return


def remove_replication_general_configs():
    try:
        click.secho("removing replication general config", fg='green')
        p = Popen(["sh", mapper.get('GeneralScripts', 'removeReplicationGeneralConfigurations')], stdout=PIPE,
                  stderr=PIPE)

        return

    except Exception as e:
        print(e)
        return


def get_redundancy_nodelist():
    try:

        p = Popen(["sh", mapper.get('GeneralScripts', 'getNodeList')], stdout=PIPE, stderr=PIPE)
        lines = []
        for line in p.stdout:
            line = line.decode("utf-8")
            line.replace(" ", "")
            line.replace("\t", "")
            line.replace("\n", "")
            if len(line) > 1:
                lines.append(line)

        if len(lines) > 0:
            return lines[0]
        else:
            return None
    except Exception as e:
        print(e)
    return None


def get_maintenance_mode_nodelist():
    try:

        p = Popen(["sh", mapper.get('GeneralScripts', 'getMaintenanceModeNodeList')], stdout=PIPE, stderr=PIPE)
        lines = []
        for line in p.stdout:
            line = line.decode("utf-8")
            line = line.lstrip()
            line.replace("\t", "")
            line.replace("\n", "")

            if len(line) > 1:
                lines = line.split(" ")

        if len(lines) > 0:
            return lines
        else:
            return []
    except Exception as e:
        print(e)

    return []


def get_redundancy_offline():
    try:

        p = Popen(["sh", mapper.get('GeneralScripts', 'getOfflineNode')], stdout=PIPE, stderr=PIPE)
        lines = []
        for line in p.stdout:
            line = line.decode("utf-8")
            lines.append(line)

        if len(lines) > 0:
            second = lines[0]
            if reach_ip(second):
                return None
            else:
                return second
        else:
            return None

    except Exception as e:
        print(e)
    return ""


def get_redundancy_online():
    try:

        p = Popen(["sh", mapper.get('GeneralScripts', 'getOfflineNode')], stdout=PIPE, stderr=PIPE)
        lines = []
        for line in p.stdout:
            line = line.decode("utf-8")
            lines.append(line)

        second = lines[0]
        node_list = get_redundancy_nodelist()
        if reach_ip(second):
            return node_list
        else:
            return node_list.replace(second, "")

    except Exception as e:
        print(e)
    return ""


def get_virtual_ip():
    try:

        p = Popen(["sh", mapper.get('GeneralScripts', 'getVirtualIp')], stdout=PIPE, stderr=PIPE)
        lines = []
        for line in p.stdout:
            line = line.decode("utf-8")
            lines.append(line)

        result = lines[0]

        result = result.split('=')

        return result[1]

    except Exception as e:
        print(e)
    return ""


def force_master():
    if click.confirm('This action can affect external Load Balancer, Are you sure you want to continue?'):

        p = Popen(["sh", mapper.get('GeneralScripts', 'forceMaster')], stdout=PIPE, stderr=PIPE)
        for line in p.stdout:
            line = line.decode("utf-8")
            print(line)
        p.wait()

        click.pause('Master Node Set Correctly, please press any key to continue...')


def galeraReplicationInfo():
    p = Popen(["sh", mapper.get('GeneralScripts', 'checkGaleraStatus')], stdout=PIPE, stderr=PIPE)
    r = ""
    for line in p.stdout:
        line = line.decode("utf-8")
        r = line
    p.wait()

    return r


def show_redundancy_status():
    try:
        p = Popen(["sh", mapper.get('GeneralScripts', 'isRedundancyEnabled')], stdout=PIPE, stderr=PIPE)

        lines = []
        for line in p.stdout:
            line = line.decode("utf-8")
            lines.append(line)

        time.sleep(2)

        p = Popen(["sh", mapper.get('GeneralScripts', 'getNodeList')], stdout=PIPE, stderr=PIPE)

        node_list = []
        for line in p.stdout:
            line = line.decode("utf-8")
            node_list.append(line)

        if len(lines) > 1:
            if "Slave_IO_Running: Yes" in lines[1]:
                print("Status: Master-Master Database Replication is Running \n Nodes:" + node_list[0])
            else:
                print("Status: Master-Master Database Replication is Not Running")
        else:
            print("Status: Master-Master Database Replication is Not Running")

        click.pause("Press any key to continue...")

    except Exception as e:
        print(e)
        return


def createHA(ctx, installed_app):
    # Checking if redundancy is enable
    node_list = get_redundancy_nodelist()
    is_redundancy_on = (node_list != None and ' '.join(
        node_list.split()) != '') or is_mastermaster_mastersalve_replication_enabled()
    ip = get_simple_hostname().replace("\n", "")
    click.clear()
    ha_nodes = []

    if is_redundancy_on:
        click.secho("Node List: " + node_list, fg='green')
        ha_nodes = node_list.split(" ")
        ha_nodes = ha_nodes[1:]

    else:
        click.secho('\nPlease follow the steps to configure High Availability', fg='green')
        click.secho('Node 1: %s' % get_simple_hostname(), fg='green')
        value = click.prompt('Please enter the second hostname.\nNode 2', type=str).replace(
            "\n", "")
        while not reach_ip(value):
            click.pause('Node is not reachable, please enter any key to continue...')
            value = click.prompt('Please enter the second hostname.\nNode 2',
                                 type=str).replace("\n", "")

        ha_nodes += [value]

        confirm = True
        nodes_numbers = 2
        max_nodes_number = 2

        if installed_app == "aco":
            max_nodes_number = 4
        else:
            max_nodes_number = 9

        while (nodes_numbers < max_nodes_number and confirm):
            confirm = click.confirm('Do you want to add another Node?')
            if confirm:
                value = click.prompt(
                    'Node %s Please enter the hostname.' % str(nodes_numbers + 1),
                    type=str).replace("\n", "")
                while not reach_ip(value):
                    click.pause('Node is not reachable, please enter any key to continue...')
                    value = click.prompt(
                        'Node %s Please enter the hostname.' % str(
                            nodes_numbers + 1), type=str).replace("\n", "")

                ha_nodes += [value]
            nodes_numbers += 1

    if len([ip] + ha_nodes) == 2 and installed_app != 'aco':
        replication_type = "master-master"
        print('replication is master-master')
    else:
        replication_type = "master-slaves"
        print('replication is master-slaves')

    config_ha = True
    if not is_redundancy_on:
        # if installed_app != 'aco':
        config_ha = not click.confirm('Will you be using an External Load Balancer to route all traffic to the nodes?')

    if config_ha:
        virtual_ip = click.prompt('Virtual IP', type=str).replace("\n", "")
        virtual_ip_mask = click.prompt('Virtual IP Mask', type=str).replace("\n", "")

        password = click.prompt('Cluster Password', type=str, hide_input=True, confirmation_prompt=True)
        while '\\' in password or '/' in password or '$' in password:
            click.pause('Password cannot have \\ nor / nor $, please press any key to continue')
            password = click.prompt('Cluster Password', type=str, hide_input=True, confirmation_prompt=True)

        click.secho('\nChanges to be saved:', fg='green')
        click.secho('Master node: %s' % ip, fg='green')
        for i in range(len(ha_nodes)):
            click.secho('Secondary node %s' % str(i + 1) + ': ' + ha_nodes[i], fg='green')
        click.secho('Virtual IP: %s' % virtual_ip, fg='green')
        click.secho('Virtual IP Mask: %s' % virtual_ip_mask, fg='green')

        if click.confirm('Do you want to save changes and enable High Availability?'):
            print("Please wait while High Availability is configured and enabled...")

            nodes = ""
            for i in ha_nodes:
                nodes += str(i) + " "

            p = Popen(
                ["sh", mapper.get('GeneralScripts', 'haInstall'), ip, nodes, virtual_ip, virtual_ip_mask, password],
                stdout=PIPE, stderr=PIPE)

            out, err = p.communicate()
            time.sleep(10)

            for line in out.splitlines():
                print(line)

            for line in err.splitlines():
                print(line)

                if 'ERROR' in line:
                    click.secho(line, fg='red')
                    click.secho(
                        'There was an error setting up HA. Check HA status and ensure hostnames are unique. You can validate this, using option 7 in Network menu')
                    click.secho(
                        'After ensuring hostnames are unique, you will need to remove cluster configurations. You can do this, using option 5 in High Availability menu')
                    click.pause('Press any key to continue...')
                    return 'ok'

            print('Please wait while High Availability is enabled...')
            time.sleep(20)

    if not is_redundancy_on:
        if installed_app == 'aco':

            print('-- Adding Mongo Replica!!!')
            if not config_ha:  # using external lb
                ha_nodes = ha_nodes.insert(ip, 0)
                ctx.invoke(write_nodes, ha_nodes=ha_nodes)

            ctx.invoke(add_mongo_replication)
            ctx.invoke(mongo_replication_postconditions)
            print('Ending mongo replication')

        elif replication_type == "master-slaves":
            print('-- Adding Database Replication Settings')
            print('-- Resetting Synchronization Settings')
            remove_replication_general_configs()
            print('-- General Configs were removed')
            remove_replication_db_configs()
            print('-- Preparing Database Server for Replication')
            add_replication_preconditions()
            print('-- Activating Database Replication')
            determine_master_and_run_replication()
            print('-- Ending Database Server Configurations')
            add_replication_posconditions()

        elif replication_type == "master-master":
            ha_nodes = [ip] + ha_nodes
            ctx.invoke(enable_redundancy, ha_nodes=ha_nodes)
            menu = 'main'

        click.pause('Press any key to continue...')

    return 'ok'


##################### Refactoring #######################################

def is_db_replication_set_in_mongo():
    return boolean_evaluator('isMongoConfiguredThroughFile')


# I can go deeper and check mariadb configurations
def is_db_replication_set_in_mariadb():
    return boolean_evaluator('isRedundancyEnableMariadb')


# I can go deeper and check mariadb configurations
def is_db_replication_set_in_galera():
    return boolean_evaluator('isRedundancyEnableGalera')


def are_pcs_pacemaker_corosync_running():
    return boolean_evaluator('arePcsPacemakerCorosyncRunning')


#######################################################################################################################
def is_mastermaster_mastersalve_replication_enabled():
    try:
        p = Popen(["sh", mapper.get('GeneralScripts', 'isRedundancyEnabled')], stdout=PIPE, stderr=PIPE)

        lines = []
        for line in p.stdout:
            line = line.decode("utf-8")
            lines.append(line)

        if len(lines) > 1:
            if "Slave_IO_Running: Yes" in lines[1]:
                return True
            else:
                return False
        else:
            return False

    except Exception as e:
        print(e)
    return False


def enable_redundancy(ha_nodes):
    try:
        click.secho('Adding Master-Master Replication Pre-conditions', fg='green')
        p = subprocess.Popen(
            ["sh", mapper.get('GeneralScripts', 'enableRedundancyPreconditions'), ha_nodes[0], ha_nodes[1]],
            stdout=PIPE, stderr=PIPE)

        time.sleep(5)

        click.secho('Enabling Master-Master Replication', fg='green')
        p = subprocess.Popen(["sh", mapper.get('GeneralScripts', 'enableRedundancy')], stdout=PIPE, stderr=PIPE)

        time.sleep(2)
        click.secho('Adding Master-Master Replication Post-conditions', fg='green')
        p = subprocess.Popen(["sh", mapper.get('GeneralScripts', 'enableRedundancyPostconditions')], stdout=PIPE,
                             stderr=PIPE)
        time.sleep(2)
        click.pause(
            "Replication has been initiated, it takes about 1 minute for the changes take effect, press any key to continue...")
        # click.pause("Master-Master Replication script has been executed, please read the output carefully and press any key to continue...")

        return True

    except Exception as e:
        click.pause(e)
        return False


def replication_masterslaves_status():
    try:
        print('Getting Replication status')
        p = Popen(["sh", mapper.get('GeneralScripts', 'dbReplicationStatus')], stdout=PIPE, stderr=PIPE)

        for line in p.stdout:
            line = line.decode("utf-8")
            print(line)
        click.pause('Please press any key to continue...')
        return
    except Exception as e:
        print(e)
        return


def determine_master_and_run_replication():
    try:
        p = Popen(["sh", mapper.get('GeneralScripts', 'dbDetermineMasterAndRunMasterReplication')], stdout=PIPE,
                  stderr=PIPE)

        lines = []
        for line in p.stdout:
            line = line.decode("utf-8")
            lines.append(line)

        print("Success")
        return

    except Exception as e:
        print(e)
        return


def is_replica_enabled():
    try:
        p = subprocess.Popen(["sh", mapper.get('GeneralScripts', 'statusMongoReplication')], stdout=PIPE,
                             stderr=PIPE)

        f = open("/home/akkadianuser/scripts/mongo_replica_scripts/status_output", "r")
        contents = f.read()
        if '"ok" : 1' in contents:
            return True
        else:
            return False
    except Exception as e:
        return False


def remove_replication_db_configs():
    try:
        click.secho("removing replication db configs", fg='green')
        p = Popen(["sh", mapper.get('GeneralScripts', 'removeReplicationDbConfigurations')], stdout=PIPE, stderr=PIPE)

        # for line in p.stdout:
        # line = line.decode("utf-8")
        #    print(line)

        return
    except Exception as e:
        print(e)
        return


def write_nodes(ha_nodes):
    # ['aco-gonza1', ['aco-gonza2']]
    nodes_string = ""  # type: str
    # click.secho("nodes_string: " + nodes_string, fg='red')
    for node in ha_nodes:
        nodes_string = nodes_string + ' ' + node

    p = Popen(["sh", mapper.get('GeneralScripts', 'writeNodes'), nodes_string], stdout=PIPE, stderr=PIPE)

    lines = []
    for line in p.stdout:
        line = line.decode("utf-8")
        lines.append(line)

    return lines


def remove_resource(resource):
    p = Popen(["sh", mapper.get('GeneralScripts', 'removeResource'), resource], stdout=PIPE, stderr=PIPE)

    out, err = p.communicate()
    time.sleep(10)

    for line in out.splitlines():
        print(line)

    for line in err.splitlines():
        print(line)

        if 'Error' in line:
            click.pause('ERROR: The resource was not removed. Press any key to continue...')
            return

    click.pause('The resource was removed correctly. Press any key to continue...')


def move_resource(resource, destination):
    p = Popen(["sh", mapper.get('GeneralScripts', 'moveResource'), resource, destination], stdout=PIPE, stderr=PIPE)

    out, err = p.communicate()
    time.sleep(10)

    for line in out.splitlines():
        print(line)

    for line in err.splitlines():
        print(line)

        if 'Error' in line:
            click.pause('ERROR: The resource was not moved. Press any key to continue...')
            return

    click.pause('The resource was moved correctly. Press any key to continue...')


def add_resource():
    click.secho('\nPlease enter the follow values to configure New Resource', fg='green')
    name = click.prompt('Resource Name', type=str).replace("\n", "")

    fields = click.prompt('Standard', type=str).replace("\n", "")
    fields += ':'
    fields += click.prompt('Provider', type=str).replace("\n", "")
    fields += ':'
    fields += click.prompt('Type', type=str).replace("\n", "")

    click.secho('Please enter all options in a single word separated by "-".')
    click.secho('VIRTUAL IP Ex: ip=192.168.0.99-cidr_netmask=24-nic=eth2-op-monitor-interval=30s', fg='yellow',
                bg='red')
    options = click.prompt('Options', type=str).replace("\n", "")

    click.secho('\nChanges to be saved:', fg='green')
    click.secho('Resource Name: %s' % name, fg='green')
    click.secho('Fields: %s' % fields, fg='green')
    click.secho('Options: %s' % options, fg='green')

    if click.confirm('Do you want to save changes?'):
        print("This will delay, please wait while everything necessary is installed...")

        p = Popen(["sh", mapper.get('GeneralScripts', 'addResource'), name, fields, options], stdout=PIPE, stderr=PIPE)

        out, err = p.communicate()
        time.sleep(10)

        for line in out.splitlines():
            print(line)

        for line in err.splitlines():
            print(line)

            if 'Error' in line:
                click.pause('ERROR: The resource was not added. Press any key to continue...')
                return

        click.pause('The new resource was added correctly. Press any key to continue...')


def edit_email_configs():
    """Edit email_configs.ini file"""
    click.edit(filename="/home/akkadianuser/email_configs.ini")


def send_email():
    email_config = False
    configs = configparser.RawConfigParser()
    try:
        configs.read('/home/akkadianuser/email_configs.ini')
        email_config = True
    except:
        email_config = False

    if email_config:

        send_email = configs.get('General', 'enable')
        if send_email:
            msg = EmailMessage()

            node_list = get_redundancy_nodelist()
            offline = ""
            online = ""

            nodes_list_array = node_list.split(' ')

            for i in range(0, len(nodes_list_array)):
                if reach_ip(nodes_list_array[i]):
                    online = online + " " + nodes_list_array[i]
                else:
                    offline = offline + " " + nodes_list_array[i]

            # GET THE NAME OF THE NODE DOWN.
            message = "HA Cluster connectivity to " + offline + " has been lost, please verify cluster status in Akkadian Appliance Manager"

            msg.set_content(message)

            installed_app = installed_application()
            name_to_show = ""
            if installed_app == 'aco':
                name_to_show = "Akkadian Console Operator"
            elif installed_app == 'pme_cm':
                name_to_show = 'Akkadian Provisioning Manager'
            elif installed_app == 'acm':
                name_to_show = 'Akkadian Contact Manager'

            msg['Subject'] = f'[ALERT] Server status: DOWN - ' + name_to_show

            msg['From'] = configs.get('General', 'sender')
            msg['To'] = configs.get('General', 'receiver')

            port = configs.get('General', 'port')  # For SSL
            password = configs.get('General', 'password')

            ssl_ = configs.get('General', 'is_ssl_enable')
            # Create a secure SSL context

            if ssl_ == "True":
                context = ssl.create_default_context()
                context.check_hostname = False
                context.verify_mode = ssl.CERT_NONE

            sender_email = configs.get('General', 'sender')
            receiver_email = configs.get('General', 'receiver')
            sever_add = configs.get('General', 'server')

            with smtplib.SMTP(sever_add, port) as server:
                if ssl_ == "True":
                    server.ehlo()
                    server.starttls(context=context)
                    server.ehlo()
                    server.login(sender_email, password)
                    server.send_message(msg)
                else:
                    if password != "":
                        server.login(sender_email, password)
                    server.send_message(msg)

                server.quit()
                print("Email Sent")


def send_email_test():
    if click.confirm('Are you sure you want to send a test email?'):
        send_email()
        click.pause('Email sent correctly, please confirm...')


def show_replica_failover_status():
    cp = subprocess.run(['systemctl status replica-failover'], universal_newlines=True, stdout=subprocess.PIPE,
                        stderr=subprocess.PIPE, shell=True)

    click.secho(cp.stdout)
    click.pause('Press any key to continue...')


def restart_replica_failover_status():
    cp = subprocess.run(['systemctl restart replica-failover'], universal_newlines=True, stdout=subprocess.PIPE,
                        stderr=subprocess.PIPE, shell=True)

    click.secho(cp.stdout)
    click.pause('Press any key to continue...')


def show_replica_failover_logs():
    read_log('galera_failover_replica_service')


def edit_replica_failover_logs_configs():
    """Edit logs.ini file"""
    click.edit(filename="/home/akkadianuser/logs-config.ini")


def turn_off_replica_failover_logs():

    log_config = False
    configs = configparser.ConfigParser()
    try:
        configs.read('/home/akkadianuser/logs-config.ini')
        log_config = True
    except:
        log_config = False

    if log_config:

        configs.set("General", "galera-replication", "0")

        try:
            with open('/home/akkadianuser/logs-config.ini', 'w') as configfile:
                configs.write(configfile)
            log_config = True
        except:
            log_config = False

    if log_config:
        click.secho("Replica failover was turned off ")
    else:
        click.secho("Couldn't turn off replica failover")

    click.pause('Press any key to continue...')
    return


def turn_on_replica_failover_logs():
    # CRITICAL = 50
    # ERROR = 40
    # WARNING = 30
    # INFO = 20
    # DEBUG = 10
    # NOTSET = 0

    value = click.prompt(
        'Please enter log level: C=CRITICAL, E=ERROR, W=WARNING, I=INFO, D=DEBUG \nPress quit to exit.',
        type=str).replace("\n", "")

    while value != 'quit' and (value != "C" and value != "E" and value != "W" and value != "I" and value != "D"):
        value = click.prompt(
            'Please enter log level: C=CRITICAL, E=ERROR, W=WARNING, I=INFO, D=DEBUG \nPress quit to exit.',
            type=str).replace("\n", "")

    newvalue = "0"
    if value == "C":
        newvalue = "50"
    elif value == "E":
        newvalue = "40"
    elif value == "W":
        newvalue = "30"
    elif value == "I":
        newvalue = "20"
    elif value == "D":
        newvalue = "10"
    else:
        newvalue = "0"

    if value == 'quit':
        return

    log_config = False
    configs = configparser.ConfigParser()
    try:
        configs.read('/home/akkadianuser/logs-config.ini')
        log_config = True
    except:
        log_config = False

    if log_config:

        configs.set("General", "galera-replication", newvalue)

        try:
            with open('/home/akkadianuser/logs-config.ini', 'w') as configfile:
                configs.write(configfile)
            log_config = True
        except:
            log_config = False

    if log_config:
        click.secho("Replica failover was turned on ")
    else:
        click.secho("Couldn't turn on replica failover")

    click.pause('Press any key to continue...')
    return


def clean_replica_failover_logs():
    try:
        fo = open("/home/akkadianuser/logs/galera_failover_recovery.log", "w+")
        fo.truncate()
        fo.close()
        click.secho('The log file was cleaned', fg='green')
    except Exception as e:
        click.secho('The file does not exist', fg='green')

    click.pause('Please press any key to continue...')
    return
