#!/usr/bin/env bash

LARAVEL_UPDATER_VERSION=1.19

LRED='\033[1;31m'
LBLUE='\033[1;34m'
LGREEN='\033[1;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

#No params so show help message
if [[ $# -eq 0 ]]
then
    echo -e "\n${LGREEN}Laravel Updater v${LARAVEL_UPDATER_VERSION}${NC}"
    echo -e "${YELLOW}Usage: $0 update_file.zip 'theZipPwd' 'App Checksum Name' /path/to/updatable/app${NC}\n"
    exit 0
fi

#Check arguments
if [[ -z $1 ]]
then
    echo -e "${LRED}Missing update zip file path parameter.${NC}"
    exit 1
fi

if [[ -z $2 ]]
then
    echo -e "${LRED}Missing zip file password parameter.${NC}"
    exit 1
fi

if [[ -z $3 ]]
then
    echo -e "${LRED}Missing app checksum name parameter.${NC}"
    exit 1
fi

if [[ -z $4 ]]; then
    echo -e "${LRED}Missing app directory parameter.${NC}"
    exit 1;
fi

#sudo sh /home/akkadianuser/scripts/laravel_updater_old.sh /var/www/product_update/${product_package} 'V33NEup6YuGC2zdte2sRs' $check '/var/www/html/'$product_name

SOURCE_UPDATE_FILE_PATH=$1
ZIP_FILE_PASSWORD=$2
APP_CHECKSUM=$3 #$(echo "$3" | sha256sum | tr -d "\n")
APP_DIRECTORY=$4

bn=$(basename $1)
TEMP_UPDATE_FILE_PATH="/tmp/$bn"
TEMP_UPDATE_DIRECTORY="/tmp/akkadian_updator/$(uuidgen)"

APP_GIT_CMD="git --work-tree=$APP_DIRECTORY --git-dir=$APP_DIRECTORY/.git"
ARTISAN_CMD="php $APP_DIRECTORY/artisan"

findHighestVersion () {
    #param is read into an array as tokens separated by dot
    read -a VERSION_A_NUMBERS <<< ${1//./$' '}
    read -a VERSION_B_NUMBERS <<< ${2//./$' '}

    let VERSION_A_NUMBERS_LENGTH=${#VERSION_A_NUMBERS[*]}
    let VERSION_B_NUMBERS_LENGTH=${#VERSION_B_NUMBERS[*]}

    if [[ $VERSION_A_NUMBERS_LENGTH > $VERSION_B_NUMBERS_LENGTH ]]
    then
        let MAX_VERSION_INDEX=$VERSION_A_NUMBERS_LENGTH-1
    else
        let MAX_VERSION_INDEX=$VERSION_B_NUMBERS_LENGTH-1
    fi

    #Find highest version
    for ((i = 0; i <= MAX_VERSION_INDEX; ++i))
    do
        #Get value, convert to number using let, default to 0 if index undefined
        let NUMBER_A=${VERSION_A_NUMBERS[$i]}
        let NUMBER_B=${VERSION_B_NUMBERS[$i]}

        if [[ $NUMBER_B > $NUMBER_A ]]
        then
          return 1
        fi

        if [[ $NUMBER_B < $NUMBER_A ]]
        then
          return 0
        fi
    done

    return 0
}

#Remove origin to prevent someone from manually doing a fetch and messing things up
echo -e "${LBLUE}Removing remote origin...${NC}"
$APP_GIT_CMD remote rm origin

#Make sure we don't start with stale data
echo -e "${LBLUE}Setting up temp dir '${YELLOW}$TEMP_UPDATE_DIRECTORY${NC}${LBLUE}'${NC}"
rm -rf "$TEMP_UPDATE_DIRECTORY"
rm -f "$TEMP_UPDATE_FILE_PATH"
mkdir -p "$TEMP_UPDATE_DIRECTORY"

#If we need to download the file
if [[ ( $SOURCE_UPDATE_FILE_PATH == "http"* ) || ( $SOURCE_UPDATE_FILE_PATH == "ftp"* ) ]]
then
    #Download update file to working directory
    echo -e "${LBLUE}Downloading source file...${NC}"
    wget "$SOURCE_UPDATE_FILE_PATH" -O "$TEMP_UPDATE_FILE_PATH"
else
    #Copy update file to working directory
    sudo cp "$SOURCE_UPDATE_FILE_PATH" "$TEMP_UPDATE_FILE_PATH"
fi

#Unzip
echo -e "${LBLUE}Unzipping update file...${NC}'"
unzip -o -P "$ZIP_FILE_PASSWORD" "$TEMP_UPDATE_FILE_PATH" -d "$TEMP_UPDATE_DIRECTORY"
VALIDATOR_FILE_NAME=project_validator.txt
BUNDLE_FILE_NAME=$(ls "$TEMP_UPDATE_DIRECTORY/"*.bundle | tail -1)

echo "$BUNDLE_FILE_NAME"
BUNDLE_FILE_NAME=$(basename "$BUNDLE_FILE_NAME")

#Verify validator file exists
if [[ ! -f $TEMP_UPDATE_DIRECTORY/$VALIDATOR_FILE_NAME ]]
then
    echo -e "${LRED}Update zip file is missing validator file.${NC}"
    exit 1
fi

#Verify bundle file exists
if [[ ! -f $TEMP_UPDATE_DIRECTORY/$BUNDLE_FILE_NAME ]]
then
    echo -e "${LRED}Update zip file is missing bundle file.${NC}"
    exit 1
fi

#Validate checksum
VALIDATOR_FILE_CHECKSUM=$(cat "$TEMP_UPDATE_DIRECTORY/$VALIDATOR_FILE_NAME" | tr -d "\n")
if [[ $VALIDATOR_FILE_CHECKSUM != $APP_CHECKSUM ]]
then
    echo -e "${LRED}Invalid update file. Checksum validation failed.${NC}"
    exit 1
fi

#Verify bundle
VERIFY_BUNDLE_RESPONSE=$($APP_GIT_CMD bundle verify "$TEMP_UPDATE_DIRECTORY/$BUNDLE_FILE_NAME" | tail -1 | tr -d "\n")
if [[ $VERIFY_BUNDLE_RESPONSE != 'The bundle records a complete history.' ]]
then
    echo -e "${LRED}Unable to verify bundle.${NC}";
    exit 1;
fi

#Get update file version from bundle file (release/x.x.x.x)
UPDATE_RELEASE_VERSION_FULL_TAG=$($APP_GIT_CMD bundle list-heads "$TEMP_UPDATE_DIRECTORY/$BUNDLE_FILE_NAME" | grep -Po "release/\d+\.\d+\.\d+\.\d+" | sort -V | tail -1)

#Remove the branch prefix: turn release/1.0.26.1 to 1.0.26.1
UPDATE_RELEASE_VERSION=$(basename "$UPDATE_RELEASE_VERSION_FULL_TAG")

#Get current local version
INSTALLED_RELEASE_VERSION_FULL_TAG=$($APP_GIT_CMD describe --tags --exact-match)

#Remove the branch prefix: turn release/1.0.26.1 to 1.0.26.1
INSTALLED_RELEASE_VERSION=$(basename "$INSTALLED_RELEASE_VERSION_FULL_TAG")

if [[ ! $INSTALLED_RELEASE_VERSION ]]
then
    echo -e "${LRED}Unable to detect installed version.${NC}";
    exit 1;
fi

if [[ ! $UPDATE_RELEASE_VERSION ]]
then
    echo -e "${LRED}Unable to detect update version.${NC}";
    exit 1;
fi

echo -e "Installed version: ${LGREEN}${INSTALLED_RELEASE_VERSION}${NC}"
echo -e "Update version: ${LGREEN}${UPDATE_RELEASE_VERSION}${NC}"

d=$(date)
app_name=$(basename $4)
echo "$app_name,$d,${INSTALLED_RELEASE_VERSION},${UPDATE_RELEASE_VERSION}" >> /var/www/product_update/installation_history.csv

#Abort if update version <= current installed version
findHighestVersion $UPDATE_RELEASE_VERSION $INSTALLED_RELEASE_VERSION
[[ $? = 0 ]] && HIGHEST_VERSION=$UPDATE_RELEASE_VERSION || HIGHEST_VERSION=$INSTALLED_RELEASE_VERSION

if [[ $UPDATE_RELEASE_VERSION == $INSTALLED_RELEASE_VERSION || $HIGHEST_VERSION != $UPDATE_RELEASE_VERSION ]]
then
    echo -e "${YELLOW}There is no update available.${NC}"
    exit 0
fi

# Delete all tags except current tag
echo -e "${LBLUE}Cleaning up old tags...${NC}"
$APP_GIT_CMD tag -d $($APP_GIT_CMD tag -l | grep -v "^$INSTALLED_RELEASE_VERSION_FULL_TAG\$")

#Unbundle
echo -e "${LBLUE}Unbundling...${NC}"
$APP_GIT_CMD bundle unbundle "$TEMP_UPDATE_DIRECTORY/$BUNDLE_FILE_NAME"
$APP_GIT_CMD fetch --no-tags -f "$TEMP_UPDATE_DIRECTORY/$BUNDLE_FILE_NAME" "refs/tags/$UPDATE_RELEASE_VERSION_FULL_TAG:refs/tags/$UPDATE_RELEASE_VERSION_FULL_TAG"

#Checking out release update
echo -e "${LBLUE}Checking out version...${NC}"
$APP_GIT_CMD checkout -f "$UPDATE_RELEASE_VERSION_FULL_TAG"

#Update permissions
echo -e "${LBLUE}Updating Permissions...${NC}"
chown -R root:apache "$APP_DIRECTORY"
chmod -R 2775 "$APP_DIRECTORY"
chmod -R 2777 "$APP_DIRECTORY/storage"

#Clear cache
echo -e "${LBLUE}Clearing Cache...${NC}"
rm -f "$APP_DIRECTORY/bootstrap/cache/"*.php
$ARTISAN_CMD cache:clear
$ARTISAN_CMD view:clear
$ARTISAN_CMD config:clear
$ARTISAN_CMD route:clear

#Run migrations
echo -e "${LBLUE}Running migrations...${NC}"
$ARTISAN_CMD migrate --force

#Restart queue
echo -e "${LBLUE}Restarting queue...${NC}"
$ARTISAN_CMD queue:restart

#Optimize
echo -e "${LBLUE}Optimizing Laravel...${NC}"
$ARTISAN_CMD optimize

#Make sure we don't count permission differences as a change in git
echo -e "${LBLUE}Updating file mode...${NC}"
$APP_GIT_CMD config core.fileMode false

#Update permissions again in case migrations or
#queue restart commands modified/created files as current logged in user
echo -e "${LBLUE}Finalizing Permissions...${NC}"
chown -R root:apache "$APP_DIRECTORY"
chmod -R 2775 "$APP_DIRECTORY"
chmod -R 2777 "$APP_DIRECTORY/storage"

#Cleanup
echo -e "${LBLUE}Cleaning up...${NC}"
rm -rf "$TEMP_UPDATE_DIRECTORY"
rm -f "$TEMP_UPDATE_FILE_PATH"

echo -e "${LGREEN}Done.${NC}"