from twisted.internet.protocol import Factory
from twisted.protocols.basic import LineReceiver
import sched, time
from decimal import Decimal
import threading
import logging
import configparser
from ReplicaFailover import ReplicaFailover

logconfig = configparser.RawConfigParser()
logconfig.read('/home/akkadianuser/logs-config.ini')
loglevel = logconfig.get('General', 'galera-replication')

log_level = int(loglevel)
logging.basicConfig(format='%(asctime)s %(levelname)-8s %(message)s', filename='/home/akkadianuser/logs/galera_failover_recovery.log', level=log_level, datefmt='%Y-%m-%d %H:%M:%S')
KEEP_ALIVE_SECONDS = 10


class FailoverServer(LineReceiver, ReplicaFailover):

    def __init__(self, users):
        self.__client_alive = 0
        self.__scheduler = sched.scheduler(time.time, time.sleep)
        self.__lastDataReceivedTime = time.time()
        self.__t = None

    def connectionMade(self):
        logging.debug("Server Connection made")
        self.__peer = self.transport.getPeer()
        self.__client_alive = self.__client_alive + 1

        self.__scheduler.enter(KEEP_ALIVE_SECONDS, 1, self.sendKeepAlive)
        self.__t = threading.Thread(target=self.__scheduler.run)
        self.__t.start()

        self.__handleConnectionMade()

    def sendKeepAlive(self):
        self.checkDataReceived()
        self.sendLine("Are you alive?".encode("utf-8"))

        if self.__client_alive > 0:
            self.__scheduler.enter(KEEP_ALIVE_SECONDS, 1, self.sendKeepAlive)

        logging.debug("The keep alive mechanism was setup")

    def checkDataReceived(self):
        this_moment = time.time()
        a = Decimal(this_moment)
        b = Decimal(self.__lastDataReceivedTime)
        c = a - b

        if c > (KEEP_ALIVE_SECONDS * 2) + 5:

            logging.debug("The client did not respond... Triggering lost connection")
            # stopping the thread!
            self.__client_alive = self.__client_alive - 1
            self.__handleConnectionLost()
        else:
            logging.debug("The connection with the client was good")

    def dataReceived(self, data):
        if data.decode("utf-8").strip().find("alive:yes") != -1:
            logging.debug("Client is alive")
            self.__lastDataReceivedTime = time.time()
        else:
            self.__client_alive = self.__client_alive - 1
            logging.debug("Client is not alive")

    def connectionLost(self, reason):
        logging.debug("Connection lost")
        self.__handleConnectionLost()

    def lineReceived(self, line):
        if line.decode() == "alive:yes":
            print("Client is alive")
            self.sendLine("Are you alive?")

    def __handleConnectionLost(self):
        logging.debug("Handling Connection lost in server")
        logging.debug('In server, client connection lost detected, trying to handle it')

        # this is hardcoded with aPM and aCM user/password change it!!!
        logging.debug('Recovering mariadb one node')
        ReplicaFailover.recover_mariadb_one_node(self)
        # send email to notify the node down
        ReplicaFailover.send_notify_node_down(self)

    def __handleConnectionMade(self):
        logging.debug("Handling Connection made in server")
        logging.debug('In server, client connection detected, trying to handle it')

        ReplicaFailover.connected_made_on_server(self, host=self.__peer.host)
        try:
            ReplicaFailover.cancel_notify_node_down(self)
        except ValueError as ve:
            logging.debug('Trying to cancel unexisting event: in_server_client_connection_established')
        except Exception as e:
            logging.debug('Exception cancelind')


class ReplicaFailoverServerFactory(Factory):
    def __init__(self):
        self.users = {}  # maps user names to Chat instances

    def buildProtocol(self, addr):
        return FailoverServer(self.users)

