from twisted.internet.protocol import Protocol, ReconnectingClientFactory
from twisted.protocols.basic import LineReceiver
import threading
import sched, time
from decimal import Decimal
import logging
import configparser
from ReplicaFailover import ReplicaFailover

logconfig = configparser.RawConfigParser()
logconfig.read('/home/akkadianuser/logs-config.ini')
loglevel = logconfig.get('General', 'galera-replication')

log_level = int(loglevel)
logging.basicConfig(format='%(asctime)s %(levelname)-8s %(message)s', filename='/home/akkadianuser/logs/galera_failover_recovery.log', level=log_level, datefmt='%Y-%m-%d %H:%M:%S')
KEEP_ALIVE_SECONDS = 10
credentialsconfig = configparser.RawConfigParser()


class FailoverClient(LineReceiver):
    # _FailoverClient__factory
    def __init__(self, factory):
        self.__factory = factory
        self.__scheduler = sched.scheduler(time.time, time.sleep)
        self.__lastDataReceivedTime = time.time()
        self.__is_server_alive = True
        pass

    def connectionMade(self):
        logging.debug("In client connection made")
        self.__scheduler.enter(KEEP_ALIVE_SECONDS, 1, self.didIReceiveKeepAlive)
        self.__t = threading.Thread(target=self.__scheduler.run)
        self.__t.start()

    def dataReceived(self, data):
        logging.debug("Data received from server")
        if data.decode("utf-8").strip().find("Are you alive?") != -1:
            self.sendLine("alive:yes".encode("utf-8"))
            self.__lastDataReceivedTime = time.time()

        else:
            pass

    def didIReceiveKeepAlive(self):
        this_moment = time.time()
        a = Decimal(this_moment)
        b = Decimal(self.__lastDataReceivedTime)
        c = a - b
        if c > (KEEP_ALIVE_SECONDS * 2) + 5:
            logging.debug("The server did not reach me... Triggering lost connection")
            # stopping the thread!
            self.__is_server_alive = False
            self._FailoverClient__factory._ReplicaFailoverClientFactory__connector.disconnect()

        else:
            logging.debug("The connection was good")

        if self.__is_server_alive:
            self.__scheduler.enter(KEEP_ALIVE_SECONDS, 1, self.didIReceiveKeepAlive)


class ReplicaFailoverClientFactory(ReconnectingClientFactory, ReplicaFailover):

    def __init__(self):
        self.factor = 1.7

    def clientConnectionLost(self, connector, reason):
        logging.debug("In client connection lost")
        ReconnectingClientFactory.clientConnectionLost(self, connector, reason)
        self.__handleConnectionLost(connector, reason)
        # start server

    def clientConnectionFailed(self, connector, reason):
        logging.debug("In client connection failed")
        ReconnectingClientFactory.clientConnectionFailed(self, connector,
                                                         reason)
        logging.debug('connection failed in client reason: ')

        self.__handleConnectionFailed(connector, reason)

    def __handleConnectionLost(self, connector, reason):
        logging.debug("In client handling connection lost")
        logging.debug('In client, server connection failed, trying to handle it')
        ReplicaFailover.recover_mariadb_one_node(self)

    def __handleConnectionFailed(self, connector, reason):
        logging.debug("In client handling connection failed")
        ReplicaFailover.recover_mariadb_one_node(self)

    def startedConnecting(self, connector):
        logging.debug("In client connection started")
        self.__connector = connector

    def buildProtocol(self, addr):
        logging.debug('In client Connected.')

        logging.debug('In client, server connection established')
        try:
            # The server will be in charge of recovery, so cancel the email notify
            ReplicaFailover.cancel_notify_node_down(self)
            logging.debug('Trying to connect in replica failover')
            ReplicaFailover.connected_made_on_client(self)


        except ValueError as ve:
            logging.debug('Trying to cancel unexisting event')

        self.resetDelay()
        return FailoverClient(self)

