import logging.config
import os
import time
from os import path
from typing import Container

import petl as etl
from petl.transform import validation
from watchdog.events import FileSystemEventHandler

from cdr_db_handler import CDRDBHandler

#logging.basicConfig(filename='myapp.log', level=logging.INFO, format='%(asctime)s %(levelname)s AkkadianCDRCollector: %(message)s', datefmt='%Y-%m-%dT%H:%M:%S%z')
log_file_path = path.join(path.dirname(path.abspath(__file__)), 'logging.ini')
logging.config.fileConfig(log_file_path, disable_existing_loggers=False)
logger = logging.getLogger(__name__)


class CDRFileHandler(FileSystemEventHandler):
    expectedHeaders = [
        "cdrrecordtype",
        "globalcallid_callmanagerid",
        "globalcallid_callid",
        "origlegcallidentifier",
        "datetimeorigination",
        "orignodeid",
        "origspan",
        "origipaddr",
        "callingpartynumber",
        "callingpartyunicodeloginuserid",
        "origcause_location",
        "origcause_value",
        "origprecedencelevel",
        "origmediatransportaddress_ip",
        "origmediatransportaddress_port",
        "origmediacap_payloadcapability",
        "origmediacap_maxframesperpacket",
        "origmediacap_g723bitrate",
        "origvideocap_codec",
        "origvideocap_bandwidth",
        "origvideocap_resolution",
        "origvideotransportaddress_ip",
        "origvideotransportaddress_port",
        "origrsvpaudiostat",
        "origrsvpvideostat",
        "destlegcallidentifier",
        "destnodeid",
        "destspan",
        "destipaddr",
        "originalcalledpartynumber",
        "finalcalledpartynumber",
        "finalcalledpartyunicodeloginuserid",
        "destcause_location",
        "destcause_value",
        "destprecedencelevel",
        "destmediatransportaddress_ip",
        "destmediatransportaddress_port",
        "destmediacap_payloadcapability",
        "destmediacap_maxframesperpacket",
        "destmediacap_g723bitrate",
        "destvideocap_codec",
        "destvideocap_bandwidth",
        "destvideocap_resolution",
        "destvideotransportaddress_ip",
        "destvideotransportaddress_port",
        "destrsvpaudiostat",
        "destrsvpvideostat",
        "datetimeconnect",
        "datetimedisconnect",
        "lastredirectdn",
        "pkid",
        "originalcalledpartynumberpartition",
        "callingpartynumberpartition",
        "finalcalledpartynumberpartition",
        "lastredirectdnpartition",
        "duration",
        "origdevicename",
        "destdevicename",
        "origcallterminationonbehalfof",
        "destcallterminationonbehalfof",
        "origcalledpartyredirectonbehalfof",
        "lastredirectredirectonbehalfof",
        "origcalledpartyredirectreason",
        "lastredirectredirectreason",
        "destconversationid",
        "globalcallid_clusterid",
        "joinonbehalfof",
        "comment",
        "authcodedescription",
        "authorizationlevel",
        "clientmattercode",
        "origdtmfmethod",
        "destdtmfmethod",
        "callsecuredstatus",
        "origconversationid",
        "origmediacap_bandwidth",
        "destmediacap_bandwidth",
        "authorizationcodevalue",
        "outpulsedcallingpartynumber",
        "outpulsedcalledpartynumber",
        "origipv4v6addr",
        "destipv4v6addr",
        "origvideocap_codec_channel2",
        "origvideocap_bandwidth_channel2",
        "origvideocap_resolution_channel2",
        "origvideotransportaddress_ip_channel2",
        "origvideotransportaddress_port_channel2",
        "origvideochannel_role_channel2",
        "destvideocap_codec_channel2",
        "destvideocap_bandwidth_channel2",
        "destvideocap_resolution_channel2",
        "destvideotransportaddress_ip_channel2",
        "destvideotransportaddress_port_channel2",
        "destvideochannel_role_channel2",
        "incomingprotocolid",
        "incomingprotocolcallref",
        "outgoingprotocolid",
        "outgoingprotocolcallref",
        "currentroutingreason",
        "origroutingreason",
        "lastredirectingroutingreason",
        "huntpilotpartition",
        "huntpilotdn",
        "calledpartypatternusage",
        "incomingicid",
        "incomingorigioi",
        "incomingtermioi",
        "outgoingicid",
        "outgoingorigioi",
        "outgoingtermioi",
        "outpulsedoriginalcalledpartynumber",
        "outpulsedlastredirectingnumber",
        "wascallqueued",
        "totalwaittimeinqueue",
        "callingpartynumber_uri",
        "originalcalledpartynumber_uri",
        "finalcalledpartynumber_uri",
        "lastredirectdn_uri",
        "mobilecallingpartynumber",
        "finalmobilecalledpartynumber",
        "origmobiledevicename",
        "destmobiledevicename",
        "origmobilecallduration",
        "destmobilecallduration",
        "mobilecalltype",
        "originalcalledpartypattern",
        "finalcalledpartypattern",
        "lastredirectingpartypattern",
        "huntpilotpattern"
    ] #list of cdr data headers from cucm 11 (min supported version)

    # event format
    # event_type — The type of the event as a string. Default to None.
    # is_directory — True if event was emitted for a directory; False otherwise.
    # src_path — Source path of the file system object that triggered this event.

    def on_created(self, event):
        try:
            self.wait_for_stable_file(event.src_path)
            valid = self.validate_cdr(event.src_path)

            if valid == True:
                logger.info("Data in "+ event.src_path+ " is valid.")
                # persist to DB
                self.store_cdr_to_db(event)
            else:
                logger.info("Data validation failed. Check that your data is correctly formatted and has all necessary headers.")
        except Exception as e:
            message = "An error occured: "+f"{type(e).__name__} at line {e.__traceback__.tb_lineno} of {__file__}: {e}"
            logger.critical(message)
            # logger.critical(
            #     f"{type(e).__name__} at line {e.__traceback__.tb_lineno} of {__file__}: {e}"
            # )
            #exit(1)

    def validate_cdr(self, filePath):
        """Validates a given CDR file.

        Args:
            filePath (string): The full path to the cdr file

        Returns:
            True: If the cdr file is valid
            False: If the cdr file isn\'t valid
        """
        try:
            logger.info("Validating: "+ filePath)
            cucm12Headers = ["origdevicetype", "destdevicetype","origdevicesessionid","destdevicesessionid"]
            validationLocation = "/usr/local/sbin/akk_cdr/validation_results.log"
            filename = os.path.basename(filePath)
            parts = filename.split("_")
            config = CDRDBHandler.loadConfig()
            strictMode = config['header_validation']
            if (parts[0] == 'cmr') & (config['enable_cmr'] == True):
                strictMode = False
            elif (parts[0] == 'cmr') & (config['enable_cmr'] == False):
                return False # bounce back 

            if len(parts) == 5:  # checking that file is named in correct format
                if strictMode == True:
                    
                    # Header operations and checks
                    initCSVData = etl.fromcsv(filePath)
                    initActualHeaders = etl.header(initCSVData)

                    # normalise headers by converting first character to lowercase. Validations fail for differing cases
                    modHeaders = []
                    for header in initActualHeaders:
                        header = header.lower()
                        modHeaders.append(header)
                    
                    csvData = etl.setheader(initCSVData, modHeaders) # replace headers with lower case versions
                    actualHeaders = etl.header(csvData) # get headers from data

                    # check if the data is CUCM 12+ and add extra headers
                    if set(cucm12Headers).issubset(set(actualHeaders)) and cucm12Headers[0] not in self.expectedHeaders:
                        # add these headers to expected headers, in order to pass validation
                        self.expectedHeaders += cucm12Headers

                    # add headers that could have been manipulated back to data.
                    #finalData =  None
                    if "destlegcallidentifier" not in actualHeaders and "destlegidentifier" in actualHeaders and "destlegidentifier" not in self.expectedHeaders:
                        #logger.info("Adding field to data table")
                        moddedCSVData = etl.addfield(csvData,"destlegcallidentifier", "0")
                        actualHeaders = etl.header(moddedCSVData)
                        moddedHeaders = self.expectedHeaders.copy()
                        moddedHeaders.append("destlegidentifier")
                        constraint = [dict(name="__header__", assertion= lambda value: value not in self.expectedHeaders)]
                        # validationResults = etl.validate(moddedCSVData, header=moddedHeaders)
                        validationResults = etl.validate(moddedCSVData, constraints=constraint)
                        actualHeaders = [] #reset variable. 
                    else:
                        #logger.info("Not adding field to data table")
                        validationResults = etl.validate(csvData, header=self.expectedHeaders)

                    if etl.nrows(validationResults) > 0:
                        # logger.info validation issues to file.
                        presentTime = time.time()
                        timeObj = time.localtime(presentTime)
                        timeStr = f"{timeObj.tm_mday}-{timeObj.tm_mon}-{timeObj.tm_year} {timeObj.tm_hour}:{timeObj.tm_min}:{timeObj.tm_sec}" # DD-MM-YYYY H:M:S
                        heading = "\n================================================================================\n"
                        heading += f"Validation results for {filePath}, generated at {timeStr}\n"
                        heading += f"Expected header count: {len(self.expectedHeaders)} \n"
                        heading += f"Header Count from data: {len(actualHeaders)}\n"
                        heading += f"Missing headers: {set(self.expectedHeaders) - set(actualHeaders) }\n"
                        heading += f"Unexpected headers: {set(actualHeaders) - set(self.expectedHeaders)}\n"
                        heading += f"Please follow the CDR header naming conventions found in the Call Detail Records Administration Guide for Cisco Unified Communications Manager v12.5, \nhttps://www.cisco.com/c/en/us/td/docs/voice_ip_comm/cucm/service/12_5_1/cdrdef/cucm_b_cdr-admin-guide-1251/cucm_b_cdr-admin-guide-1251_chapter_0101.html \n"
                        heading += "================================================================================\n"
                        
                        with open(f"{validationLocation}","a") as f:
                            data = str(validationResults.lookall())
                            f.write(heading)
                            f.write(data)
                        logger.error(f"Validation failed. Error results written to validation log at {validationLocation}")
                        return False
                else:
                    return True   # validation done really            

            return True
        except Exception as e:
            logger.error(
                f"{type(e).__name__} at line {e.__traceback__.tb_lineno} of {__file__}: {e}"
            )
            return False

    def delete_cdr(self, fileLocation):
        try:
            if os.path.isfile(fileLocation):
                logger.info('Deleting cdr file:'+ fileLocation)
                os.remove(fileLocation)
                logger.info("File at "+ fileLocation +" successfully deleted")
                return
            else:
                logger.info('CDR at'+fileLocation+"not found. Could not be deleted.")
                return
        except Exception as err:
            logger.info('CDR at'+fileLocation+" could not be deleted.")
            # logger.info(f"{type(err).__name__} at line {err.__traceback__.tb_lineno} of {__file__}: {err}")

    def store_cdr_to_db(self, event):
        """Stores the csv data in the database. Adds to a table if it exists or creates a new table if it doesnt.

        Args:
            event (): file creation event
        """
        try:
            csvData = etl.fromcsv(event.src_path)
            config = CDRDBHandler.loadConfig()
            # parts = event.src_path.split("_")
            filename = os.path.basename(event.src_path)
            parts = filename.split("_")
            tag = parts[0]
            cluster = parts[1].lower()
            nodeID = parts[2]
            dateTime = parts[3]
            seqNumber = parts[4].split(".")[0]
            cdr_info = {
                'tag': tag.lower(),
                'cdr_datetime': dateTime, 
                'cluster': cluster,
                'nodeid': nodeID,
                'seqnumber': seqNumber
            }

            tableName = cluster + "_" + nodeID

            if tag.lower() == 'cdr':
                #check to see if cdr has been stored before
                cdr_exists = CDRDBHandler.check_records_stored(cdr_info)
                if cdr_exists == False:
                    # check and see if table exists.
                    table_exists = CDRDBHandler.check_table_exists(tableName)

                    if table_exists == False:
                        # create table
                        logger.info("Table doesnt exist: "+ tableName)
                        headers = etl.header(csvData)
                        res = CDRDBHandler.create_cdr_table(tableName, headers)
                        if res == True:
                            logger.info("Table created.")

                    # persist data
                    CDRDBHandler.store_cdr_records(csvData, tableName)
                    CDRDBHandler.add_record_to_tracking(cdr_info)
                    self.delete_cdr(event.src_path)
                    return True
                else:
                    logger.info("File at:"+event.src_path+" has already been added. Skipping")
                    return False
            elif (tag.lower() == 'cmr') & (config['enable_cmr'] == True):
                #check to see if cdr has been stored before
                tableName = 'cmr_'+ tableName
                cmr_exists = CDRDBHandler.check_records_stored(cdr_info)
                if cmr_exists == False:
                    # check and see if table exists.
                    table_exists = CDRDBHandler.check_table_exists(tableName)

                    if table_exists == False:
                        # create table
                        logger.info("Table doesnt exist: "+tableName)
                        headers = etl.header(csvData)
                        res = CDRDBHandler.create_cdr_table(tableName, headers)
                        if res == True:
                            logger.info("Table created.")

                    # persist data
                    CDRDBHandler.store_cdr_records(csvData, tableName)
                    CDRDBHandler.add_record_to_tracking(cdr_info)
                    self.delete_cdr(event.src_path)
                    return True
                else:
                    logger.info("File at:"+event.src_path+" has already been added. Skipping")
                    
                    return False
            else:
                logger.info("File imported at:"+event.src_path+" does not have a correct filename.")
                
                return False
        except Exception as err:
            logger.info(f"{type(err).__name__} at line {err.__traceback__.tb_lineno} of {__file__}: {err}")
            raise
        
                
    def wait_for_stable_file(self, fileLocation):
        """Checks that a file is not being transfered before being read. Basically waits until file size is unchanged for 5 seconds

        Args:
            fileLocation (string): the location of the file 
        """
        logger.info(f"File found at {fileLocation}")
        logger.info("Checking that file can be accessed")
        while True:
            originalFileSize = os.path.getsize(fileLocation)
            time.sleep(3)
            newfileSize = os.path.getsize(fileLocation)
            if originalFileSize == newfileSize:
                # transfer done
                break
        
        return
    
