import json
import logging.config
import os
from itertools import islice
from os import path

import mysql.connector
import petl as etl
from mysql.connector import errorcode
from petl.transform.validation import validate

log_file_path = path.join(path.dirname(path.abspath(__file__)), 'logging.ini')
logging.config.fileConfig(log_file_path, disable_existing_loggers=False)
logger = logging.getLogger(__name__)

class CDRDBHandler:

    headers = [
        "cdrrecordtype",
        "globalcallid_callmanagerid",
        "globalcallid_callid",
        "origlegcallidentifier",
        "datetimeorigination",
        "orignodeid",
        "origspan",
        "origipaddr",
        "callingpartynumber",
        "callingpartyunicodeloginuserid",
        "origcause_location",
        "origcause_value",
        "origprecedencelevel",
        "origmediatransportaddress_ip",
        "origmediatransportaddress_port",
        "origmediacap_payloadcapability",
        "origmediacap_maxframesperpacket",
        "origmediacap_g723bitrate",
        "origvideocap_codec",
        "origvideocap_bandwidth",
        "origvideocap_resolution",
        "origvideotransportaddress_ip",
        "origvideotransportaddress_port",
        "origrsvpaudiostat",
        "origrsvpvideostat",
        "destlegcallidentifier",
        "destnodeid",
        "destspan",
        "destipaddr",
        "originalcalledpartynumber",
        "finalcalledpartynumber",
        "finalcalledpartyunicodeloginuserid",
        "destcause_location",
        "destcause_value",
        "destprecedencelevel",
        "destmediatransportaddress_ip",
        "destmediatransportaddress_port",
        "destmediacap_payloadcapability",
        "destmediacap_maxframesperpacket",
        "destmediacap_g723bitrate",
        "destvideocap_codec",
        "destvideocap_bandwidth",
        "destvideocap_resolution",
        "destvideotransportaddress_ip",
        "destvideotransportaddress_port",
        "destrsvpaudiostat",
        "destrsvpvideostat",
        "datetimeconnect",
        "datetimedisconnect",
        "lastredirectdn",
        "pkid",
        "originalcalledpartynumberpartition",
        "callingpartynumberpartition",
        "finalcalledpartynumberpartition",
        "lastredirectdnpartition",
        "duration",
        "origdevicename",
        "destdevicename",
        "origcallterminationonbehalfof",
        "destcallterminationonbehalfof",
        "origcalledpartyredirectonbehalfof",
        "lastredirectredirectonbehalfof",
        "origcalledpartyredirectreason",
        "lastredirectredirectreason",
        "destconversationid",
        "globalcallid_clusterid",
        "joinonbehalfof",
        "comment",
        "authcodedescription",
        "authorizationlevel",
        "clientmattercode",
        "origdtmfmethod",
        "destdtmfmethod",
        "callsecuredstatus",
        "origconversationid",
        "origmediacap_bandwidth",
        "destmediacap_bandwidth",
        "authorizationcodevalue",
        "outpulsedcallingpartynumber",
        "outpulsedcalledpartynumber",
        "origipv4v6addr",
        "destipv4v6addr",
        "origvideocap_codec_channel2",
        "origvideocap_bandwidth_channel2",
        "origvideocap_resolution_channel2",
        "origvideotransportaddress_ip_channel2",
        "origvideotransportaddress_port_channel2",
        "origvideochannel_role_channel2",
        "destvideocap_codec_channel2",
        "destvideocap_bandwidth_channel2",
        "destvideocap_resolution_channel2",
        "destvideotransportaddress_ip_channel2",
        "destvideotransportaddress_port_channel2",
        "destvideochannel_role_channel2",
        "incomingprotocolid",
        "incomingprotocolcallref",
        "outgoingprotocolid",
        "outgoingprotocolcallref",
        "currentroutingreason",
        "origroutingreason",
        "lastredirectingroutingreason",
        "huntpilotpartition",
        "huntpilotdn",
        "calledpartypatternusage",
        "incomingicid",
        "incomingorigioi",
        "incomingtermioi",
        "outgoingicid",
        "outgoingorigioi",
        "outgoingtermioi",
        "outpulsedoriginalcalledpartynumber",
        "outpulsedlastredirectingnumber",
        "wascallqueued",
        "totalwaittimeinqueue",
        "callingpartynumber_uri",
        "originalcalledpartynumber_uri",
        "finalcalledpartynumber_uri",
        "lastredirectdn_uri",
        "mobilecallingpartynumber",
        "finalmobilecalledpartynumber",
        "origmobiledevicename",
        "destmobiledevicename",
        "origmobilecallduration",
        "destmobilecallduration",
        "mobilecalltype",
        "originalcalledpartypattern",
        "finalcalledpartypattern",
        "lastredirectingpartypattern",
        "huntpilotpattern",
        "destdevicetype",
        "origdevicetype",
        "origdevicesessionid",
        "destdevicesessionid"
    ]

    def loadConfig():
        try:
            configLocation = "/usr/local/sbin/akk_cdr/akk_cdr_config.json"
            # load config and check data
            configData = None
            with open(configLocation) as f:
                configData = json.load(f)
            if "user" not in configData or configData["user"] == "":
                raise Exception("Config format incorrect. User not set or invalid.")
            elif "password" not in configData or configData["password"] == "":
                raise Exception("Config format incorrect. Password not set or invalid.")
            elif "host" not in configData or configData["host"] == "":
                raise Exception("Config format incorrect. Host not set or invalid.")
            elif "database" not in configData or configData["database"] == "":
                raise Exception("Config format incorrect. Database not set.")
            elif "cdr_location" not in configData or configData["cdr_location"] == "":
                raise Exception("Config format incorrect. CDR Location not set or invalid.")
            elif "header_validation" not in configData or configData["header_validation"] == "" or type(configData["header_validation"]) != bool:
                raise Exception("Config format incorrect. Header Validation not set or invalid.")
            elif "enable_cmr" not in configData or configData["enable_cmr"] == "" or type(configData["enable_cmr"]) != bool:
                raise Exception("Config format incorrect. Enable CMR field not set or invalid.")
            else:
                return configData
        except IOError:
            logger.critical("Could not read config file at /usr/local/sbin/akk_cdr/akk_cdr_config.json")
            exit(1)
        except Exception as err:
            logger.critical("Loading config values failed: " + err)
            exit(1)
            #raise e

    def get_db_connection(params=None):
        """Returns a DB connection given a dictionary of connection parameters. Or False if it cant create a connection
        Args:
            params (dictionary): conection parameters
        """
        try:
            if params == None:
                # give it default values from file
                params = CDRDBHandler.loadConfig()
                del params["cdr_location"]  # remove config attributes that would cause connection error
                del params["header_validation"]
                del params["enable_cmr"]
                cnx = mysql.connector.connect(**params)
                return cnx
        except mysql.connector.Error as err:
            if err.errno == errorcode.ER_ACCESS_DENIED_ERROR:
                logger.error("Database authentication failed. Please check your username or password")
            elif err.errno == errorcode.ER_BAD_DB_ERROR:
                logger.error("Database does not exist")
            else:
                logger.error("A problem was encountered while trying to connect to the database.")
                logger.error(f"{type(err).__name__} at line {err.__traceback__.tb_lineno} of {__file__}: {err}")
            exit(1)
        except Exception as e: 
            logger.error("A problem was encountered while trying to connect to the database: "+f"{type(e).__name__} at line {e.__traceback__.tb_lineno} of {__file__}: {e}")
            # logger.error(f"{type(e).__name__} at line {e.__traceback__.tb_lineno} of {__file__}: {e}")
            raise 

    def check_table_exists(table_name):
        try:
            checkTableQuery = "SELECT COUNT(*) as tableCount FROM information_schema.tables WHERE table_schema = DATABASE() AND table_name = %s"
            connection = CDRDBHandler.get_db_connection()
            cursor = connection.cursor(dictionary=True)
            cursor.execute(checkTableQuery, (table_name,))
            results = cursor.fetchall()
            cursor.close()
            connection.close()
            if results[0]["tableCount"] == 0:
                return False
            elif results[0]["tableCount"] > 0:
                return True
        except mysql.connector.Error as err:
            logger.error("A problem was encountered while working with the database.")
            logger.error(f"{type(err).__name__} at line {err.__traceback__.tb_lineno} of {__file__}: {err}")
            exit(1)
        except Exception as err:
            # raise Exception(
            #     f"{type(e).__name__} at line {e.__traceback__.tb_lineno} of {__file__}: {e}"
            # )
            logger.error("An error occured: "+f"{type(err).__name__} at line {err.__traceback__.tb_lineno} of {__file__}: {err}")
            # logger.error(f"{type(err).__name__} at line {err.__traceback__.tb_lineno} of {__file__}: {err}")
            exit(1)

        """[Adds columns to tracking table that weren't in original release. Kinds like migrations to preserve compatibility with newer versions of the collector]
        """    
    
    def run_tracking_table_upgrades():
        try:
            # check if table has the following fields and add them if missing
            connection = CDRDBHandler.get_db_connection()
            cursor = connection.cursor(dictionary=True)
            query = "SELECT COUNT(*) as columnCount FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA='akk_cdr' AND TABLE_NAME='akk_cdr_tracking' AND column_name='tag';"
            cursor.execute(query)
            results = cursor.fetchall()

            if results[0]['columnCount'] == 0:
                # create column
                createQuery="ALTER TABLE `akk_cdr_tracking` ADD COLUMN (tag text NULL)"
                cursor.execute(createQuery)

            connection.commit()
            cursor.close()
            connection.close()
        except Exception as e:
            logger.error("An error occured: "+f"{type(e).__name__} at line {e.__traceback__.tb_lineno} of {__file__}: {e}")
            raise

    def create_tracking_table():
        try:
            table_name="akk_cdr_tracking"
            connection = CDRDBHandler.get_db_connection()
            cursor = connection.cursor()
            query="""
                CREATE TABLE `akk_cdr_tracking` (
                    id int(11) AUTO_INCREMENT,
                    tag text not null,
                    cdr_datetime text not null,
                    cluster text not null,
                    nodeid text not null,
                    seqnumber text not null,
                    date_saved timestamp not null default CURRENT_TIMESTAMP,
                    PRIMARY KEY (id)
                ) ENGINE=InnoDB
            """
            cursor.execute(query)
            connection.commit()
            cursor.close()
            connection.close()
        except Exception as e:
            logger.error("An error occured: "+f"{type(e).__name__} at line {e.__traceback__.tb_lineno} of {__file__}: {e}")

    def create_cdr_table(tableName, headers):
        try:
            connection = CDRDBHandler.get_db_connection()
            cursor = connection.cursor()
            tableName = connection._cmysql.escape_string(tableName)
            tableName = tableName.decode()
            protoQuery = ""
            for item in headers:
                protoQuery += item + " TEXT, "
            createQuery = (
                """
                CREATE TABLE `"""+tableName +"""` ( 
                id int(11) AUTO_INCREMENT,
                """
                + protoQuery
                + """
                PRIMARY KEY (id) 
                ) ENGINE=InnoDB
            """
            )
            logger.info("Creating table: "+tableName)
            cursor.execute(createQuery)
            connection.commit()
            
            CDRDBHandler.run_db_index_speedup([tableName])       
            
            
            cursor.close()
            connection.close()
            return True
        except Exception as e:
            logger.error("An error occured: "+f"{type(e).__name__} at line {e.__traceback__.tb_lineno} of {__file__}: {e}")
            raise

    def store_cdr_records(csvData, table_name):
        try:
            connection = CDRDBHandler.get_db_connection()
            cursor = connection.cursor()
            config = CDRDBHandler.loadConfig()
            insertQuery = "INSERT INTO `"+table_name +"` "
            headers = CDRDBHandler.headers

            # set header validation off if dealing with a CMR file
            if table_name.startswith('cmr_'):
                config['header_validation'] = False

            protoQuery = " ("
            # create header part of query
            if config['header_validation'] == False:
                headers = etl.header(csvData)
                
            for i in range(len(headers)):
                if i == len(headers) - 1:
                    protoQuery += headers[
                        i
                    ]  # add column list without comma for last item
                else:
                    protoQuery += headers[i] + ", "  # add column list
                

            protoQuery += " ) VALUES  "

            # Create value part of query
            valueData = etl.rowslice(csvData, 1, None) # slice to remove the first row, which is filled with type info
            # csvData = etl.data(valueData)
            dataList = list(etl.data(valueData))

            # pad with empty items to match header count
            for i in range(len(dataList)):
                if len(headers) > len(dataList[i]):
                    padding = [""] * (len(headers) - len(dataList[i]))
                    tmp = list(dataList[i])
                    tmp.extend(padding)
                    dataList[i] = tuple(tmp)

            # Build VALUE part of query. Add value tuples
            for i in range(len(dataList)):
                if i == len(dataList) - 1:
                    protoQuery += str(dataList[i])
                else:
                    protoQuery += str(dataList[i]) + ", "

            # print(insertQuery+protoQuery)
            finalQuery = insertQuery + protoQuery
            # print(f"\n{finalQuery}\n")
            # with open("query.txt","a") as f:
            #     f.write(finalQuery)
                
            cursor.execute(finalQuery)
            connection.commit()
            cursor.close()
            connection.close()
            logger.info("Data saved successfully")

        except Exception as e:
            logger.error("There was an error while writing the CDR contents to the database: "+ f"{type(e).__name__} at line {e.__traceback__.tb_lineno} of {__file__}: {e}")
            raise 

    def add_record_to_tracking(cdr_info):
        try:
            connection =  CDRDBHandler.get_db_connection()
            cursor = connection.cursor()

            query="""
                INSERT INTO `akk_cdr_tracking` (
                    tag, cdr_datetime,cluster,nodeid,seqnumber
                ) VALUES (
                    %s,%s,%s,%s,%s
                )
            """
            cursor.execute(query,(cdr_info['tag'],cdr_info['cdr_datetime'], cdr_info['cluster'], cdr_info['nodeid'], cdr_info['seqnumber']))
            connection.commit()
            cursor.close()
            connection.close()
        except Exception as e:
            logger.error("An error occured: "+f"{type(e).__name__} at line {e.__traceback__.tb_lineno} of {__file__}: {e}")
            raise 
        
    def check_records_stored(cdr_info):
        try:
            connection =  CDRDBHandler.get_db_connection()
            cursor = connection.cursor()

            query="""
                SELECT COUNT(*) as row_count FROM `akk_cdr_tracking`
                WHERE 
                    tag = %s AND
                    cdr_datetime = %s AND 
                    cluster = %s AND 
                    nodeid = %s AND 
                    seqnumber = %s LIMIT 1
            """
            
            cursor.execute(query,(cdr_info['tag'], cdr_info['cdr_datetime'], cdr_info['cluster'], cdr_info['nodeid'], cdr_info['seqnumber'],))

            results = cursor.fetchall()
            
            cursor.close()
            connection.close()

            if results[0][0] == 0:
                return False
            elif results[0][0] > 0:
                return True
        except Exception as e:
            logger.error("An error occured: "+f"{type(e).__name__} at line {e.__traceback__.tb_lineno} of {__file__}: {e}")
            raise
        
    def run_db_index_speedup(tables = []):
        """If not supplied with an array of tables, gets tables in DB that do not have the index speedups specified in:  https://app.clickup.com/t/24v33z1,
         Runs the speedup queries if not present.
        """
        
        try:
            connection =  CDRDBHandler.get_db_connection()
            cursor = connection.cursor()
            
            logger.info("Running table index speedups")
            
            if len(tables) < 1:
                # Get all tables that don't have speedup
                oldTablesQuery = ("""
                    SELECT DISTINCT TABLE_NAME FROM `information_schema`.`COLUMNS`  c  WHERE c.COLUMN_NAME = "callingPartyNumber" AND c.DATA_TYPE = "TEXT"
                """)
                
                cursor.execute(oldTablesQuery)
                for TABLE_NAME in cursor:
                    tables.append(*TABLE_NAME)
                    
            if len(tables) > 0:
                
                for table in tables:
                    table = connection._cmysql.escape_string(table)
                    table = table.decode()
                    typeQuery = (""" 
                            ALTER TABLE `""" + table + """` 
                                MODIFY dateTimeOrigination INT, 
                                MODIFY duration INT,
                                MODIFY callingPartyNumber VARCHAR(255),
                                MODIFY finalCalledPartyNumber VARCHAR(255),
                                MODIFY originalCalledPartyNumber VARCHAR(255)
                            """)
                    indexQuery = ("""
                                ALTER TABLE `""" + table + """` 
                                    ADD INDEX `duration_IDX` (`duration` ASC),
                                    ADD INDEX `callingPartyNumber_IDX` (`callingPartyNumber` ASC),
                                    ADD INDEX `finalCalledPartyNumber_IDX` (`finalCalledPartyNumber` ASC),
                                    ADD INDEX `originalCalledPartyNumber_IDX` (`originalCalledPartyNumber` ASC),
                                    ADD INDEX `dateTimeOrigination_IDX` (`dateTimeOrigination` DESC)
                                """)            
                    
                    cursor.execute(typeQuery)
                    cursor.execute(indexQuery)
                    logger.info("Speedups applied.")
            else:
                logger.info("No speedups to be made. Proceeding.")
                
            connection.commit()
        
            cursor.close()
            connection.close()
            
            
        except Exception as e:
            logger.error("An error occured: "+f"{type(e).__name__} at line {e.__traceback__.tb_lineno} of {__file__}: {e}")
            raise
        